<?php
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

// Manejar preflight OPTIONS request
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

// Incluir configuración y conexión
require_once '../config.php';
require_once '../conexion.php';

try {
    // Verificar que la petición sea POST
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        throw new Exception('Método no permitido');
    }

    // Obtener datos del POST
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        throw new Exception('Datos inválidos');
    }

    $curso = $input['curso'] ?? '';
    $unidad = $input['unidad'] ?? '';
    $actividad = $input['actividad'] ?? '';

    // Validar parámetros
    if (empty($curso) || empty($unidad) || empty($actividad)) {
        throw new Exception('Parámetros requeridos: curso, unidad, actividad');
    }

    // Validar formato de curso (ej: c001)
    if (!preg_match('/^c\d{3}$/', $curso)) {
        throw new Exception('Formato de curso inválido');
    }

    // Validar formato de unidad (ej: 01)
    if (!preg_match('/^\d{2}$/', $unidad)) {
        throw new Exception('Formato de unidad inválido');
    }

    // Validar formato de actividad (ej: 1, 2, 3...)
    if (!is_numeric($actividad) || $actividad < 1) {
        throw new Exception('Formato de actividad inválido');
    }

    // Formatear número de actividad con 2 dígitos
    $actividadFormateada = sprintf('%02d', $actividad);

    // Nombre de la tabla
    $nombreTabla = "avance_{$curso}_u{$unidad}";

    // Verificar conexión a la base de datos
    if (!isset($dsn)) {
        throw new Exception('Error de conexión a la base de datos');
    }

    // Verificar si la tabla existe
    $stmt = $dsn->prepare("SHOW TABLES LIKE ?");
    $stmt->execute([$nombreTabla]);
    $tablaExiste = $stmt->rowCount() > 0;

    // Obtener todos los usuarios existentes
    $stmt = $dsn->prepare("SELECT id FROM usuarios ORDER BY id");
    $stmt->execute();
    $usuarios = $stmt->fetchAll(PDO::FETCH_COLUMN);

    if (empty($usuarios)) {
        throw new Exception('No se encontraron usuarios en la base de datos');
    }

    $totalUsuarios = count($usuarios);

    // Variables a crear/actualizar
    $variables = [
        "ACT{$actividadFormateada}" => "TINYINT(1) DEFAULT NULL COMMENT 'Estado de actividad {$actividad} (0=no completada, 1=completada)'",
        "UC" => "TINYINT(1) DEFAULT NULL COMMENT 'Unidad completada (0=no completada, 1=completada)'",
        "FIU" => "DATETIME DEFAULT NULL COMMENT 'Fecha inicio unidad'",
        "FTU" => "DATETIME DEFAULT NULL COMMENT 'Fecha término unidad'",
        "FIACT{$actividadFormateada}" => "DATETIME DEFAULT NULL COMMENT 'Fecha inicio actividad {$actividad}'",
        "FTACT{$actividadFormateada}" => "DATETIME DEFAULT NULL COMMENT 'Fecha término actividad {$actividad}'"
    ];

    $operaciones = [];

    if (!$tablaExiste) {
        // Crear la tabla
        $sql = "CREATE TABLE `{$nombreTabla}` (
            `id` int(11) NOT NULL COMMENT 'ID del usuario',
            PRIMARY KEY (`id`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci COMMENT='Tabla de avance para {$curso} unidad {$unidad}'";

        $dsn->exec($sql);
        $operaciones[] = "Tabla {$nombreTabla} creada";

        // Agregar todas las variables
        foreach ($variables as $nombre => $definicion) {
            $sql = "ALTER TABLE `{$nombreTabla}` ADD COLUMN `{$nombre}` {$definicion}";
            $dsn->exec($sql);
            $operaciones[] = "Columna {$nombre} agregada";
        }

        // Insertar todos los usuarios con valores NULL
        $placeholders = str_repeat('(?),', count($usuarios) - 1) . '(?)';
        $sql = "INSERT INTO `{$nombreTabla}` (`id`) VALUES {$placeholders}";
        $stmt = $dsn->prepare($sql);
        $stmt->execute($usuarios);
        $operaciones[] = "{$totalUsuarios} usuarios insertados con valores NULL";

    } else {
        // La tabla existe, verificar y agregar variables faltantes
        $stmt = $dsn->prepare("SHOW COLUMNS FROM `{$nombreTabla}`");
        $stmt->execute();
        $columnasExistentes = array_column($stmt->fetchAll(), 'Field');

        foreach ($variables as $nombre => $definicion) {
            if (!in_array($nombre, $columnasExistentes)) {
                $sql = "ALTER TABLE `{$nombreTabla}` ADD COLUMN `{$nombre}` {$definicion}";
                $dsn->exec($sql);
                $operaciones[] = "Columna {$nombre} agregada";

                // Inicializar valores NULL para usuarios existentes
                $sql = "UPDATE `{$nombreTabla}` SET `{$nombre}` = NULL WHERE `{$nombre}` IS NOT NULL";
                $stmt = $dsn->prepare($sql);
                $stmt->execute();
                $operaciones[] = "Columna {$nombre} inicializada con NULL";
            } else {
                $operaciones[] = "Columna {$nombre} ya existe";
            }
        }

        // Verificar si hay usuarios nuevos que agregar
        $stmt = $dsn->prepare("SELECT COUNT(*) FROM `{$nombreTabla}`");
        $stmt->execute();
        $usuariosEnTabla = $stmt->fetchColumn();

        if ($usuariosEnTabla < $totalUsuarios) {
            // Agregar usuarios faltantes
            $stmt = $dsn->prepare("SELECT id FROM `{$nombreTabla}`");
            $stmt->execute();
            $usuariosEnTabla = array_column($stmt->fetchAll(), 'id');

            $usuariosNuevos = array_diff($usuarios, $usuariosEnTabla);
            
            if (!empty($usuariosNuevos)) {
                $placeholders = str_repeat('(?),', count($usuariosNuevos) - 1) . '(?)';
                $sql = "INSERT INTO `{$nombreTabla}` (`id`) VALUES {$placeholders}";
                $stmt = $dsn->prepare($sql);
                $stmt->execute(array_values($usuariosNuevos));
                $operaciones[] = count($usuariosNuevos) . " usuarios nuevos agregados";
            }
        }
    }

    // Respuesta exitosa
    echo json_encode([
        'success' => true,
        'message' => "Tabla de avance procesada exitosamente",
        'data' => [
            'tabla' => $nombreTabla,
            'curso' => $curso,
            'unidad' => $unidad,
            'actividad' => $actividad,
            'total_usuarios' => $totalUsuarios,
            'tabla_existia' => $tablaExiste,
            'operaciones' => $operaciones
        ]
    ], JSON_UNESCAPED_UNICODE);

} catch (Exception $e) {
    // Respuesta de error
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage(),
        'error' => $e->getCode()
    ], JSON_UNESCAPED_UNICODE);
}
?>
