<?php
session_start();
header('Content-Type: application/json');

// Incluir configuración de la base de datos
require_once dirname(__FILE__) . '/../config.php';
require_once dirname(__FILE__) . '/../conexion.php';

$response = ['success' => false, 'message' => '', 'details' => []];

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $input = json_decode(file_get_contents('php://input'), true);

    $curso = $input['curso'] ?? null;
    $unidad = $input['unidad'] ?? null;
    $actividad = $input['actividad'] ?? null;
    $numero_respuesta = $input['numero_respuesta'] ?? null;

    // Validaciones básicas
    if (!$curso || !$unidad || !$actividad || !$numero_respuesta) {
        $response['message'] = 'Parámetros incompletos.';
        echo json_encode($response);
        exit;
    }

    // Normalizar formato de unidad (Unidad 01 -> 01)
    $unidad_normalizada = '';
    if (preg_match('/Unidad (\d+)/', $unidad, $matches)) {
        $unidad_normalizada = str_pad($matches[1], 2, '0', STR_PAD_LEFT);
    }

    // Asegurarse de que la actividad y número de respuesta sean números de dos dígitos
    $actividad_padded = str_pad($actividad, 2, '0', STR_PAD_LEFT);
    $numero_respuesta_padded = str_pad($numero_respuesta, 2, '0', STR_PAD_LEFT);

    $table_name = "avance_{$curso}_u{$unidad_normalizada}";
    
    // Definir las columnas a crear
    $columnas_a_crear = [
        "RL{$numero_respuesta_padded}ACT{$actividad_padded}" => "TEXT NULL DEFAULT NULL COMMENT 'Respuesta Larga del estudiante'",
        "RLE{$numero_respuesta_padded}ACT{$actividad_padded}" => "TINYINT(1) NULL DEFAULT NULL COMMENT 'Respuesta del estudiante enviada (NULL=No enviada, 1=Enviada)'",
        "RTRL{$numero_respuesta_padded}ACT{$actividad_padded}" => "TEXT NULL DEFAULT NULL COMMENT 'Retroalimentación de Respuesta Larga'",
        "NTRL{$numero_respuesta_padded}ACT{$actividad_padded}" => "DECIMAL(5,2) NULL DEFAULT NULL COMMENT 'Nota de Respuesta Larga'",
        "FRL{$numero_respuesta_padded}ACT{$actividad_padded}" => "DATETIME NULL DEFAULT NULL COMMENT 'Fecha de última actualización de respuesta del estudiante'",
        "FRTRL{$numero_respuesta_padded}ACT{$actividad_padded}" => "DATETIME NULL DEFAULT NULL COMMENT 'Fecha de retroalimentación del docente'",
        "FNTRL{$numero_respuesta_padded}ACT{$actividad_padded}" => "DATETIME NULL DEFAULT NULL COMMENT 'Fecha de registro de nota'",
        "PRL{$numero_respuesta_padded}ACT{$actividad_padded}" => "INT(11) NULL DEFAULT NULL COMMENT 'Número de palabras de la respuesta'",
        "FLRL{$numero_respuesta_padded}ACT{$actividad_padded}" => "DATETIME NULL DEFAULT NULL COMMENT 'Fecha límite para respuesta del estudiante'"
    ];

    try {
        // 1. Verificar si la tabla existe
        $stmt = $dsn->prepare("SHOW TABLES LIKE :table_name");
        $stmt->execute([':table_name' => $table_name]);
        $table_exists = $stmt->fetch() !== false;

        if (!$table_exists) {
            $response['message'] = "La tabla '{$table_name}' no existe. Primero debe crearse usando el botón 'Crear tabla' en el panel.";
            echo json_encode($response);
            exit;
        }

        $columnas_creadas = [];
        $columnas_existentes = [];
        $errores = [];

        // 2. Verificar y crear cada columna si no existe
        foreach ($columnas_a_crear as $column_name => $column_definition) {
            // Verificar si la columna existe
            $stmt_check_col = $dsn->prepare("SHOW COLUMNS FROM `{$table_name}` LIKE :col_name");
            $stmt_check_col->execute([':col_name' => $column_name]);
            $col_exists = $stmt_check_col->fetch() !== false;

            if (!$col_exists) {
                try {
                    // Agregar la columna
                    $add_column_sql = "ALTER TABLE `{$table_name}` ADD COLUMN `{$column_name}` {$column_definition}";
                    $dsn->exec($add_column_sql);
                    $columnas_creadas[] = $column_name;
                } catch (PDOException $e) {
                    $errores[] = "Error al crear columna '{$column_name}': " . $e->getMessage();
                }
            } else {
                $columnas_existentes[] = $column_name;
            }
        }

        // 3. Preparar respuesta
        if (empty($errores)) {
            $response['success'] = true;
            $response['details']['tabla'] = $table_name;
            $response['details']['columnas_creadas'] = $columnas_creadas;
            $response['details']['columnas_existentes'] = $columnas_existentes;
            
            if (!empty($columnas_creadas)) {
                $response['message'] = "✅ Se crearon " . count($columnas_creadas) . " columnas nuevas: " . implode(', ', $columnas_creadas);
                if (!empty($columnas_existentes)) {
                    $response['message'] .= " | Columnas ya existían: " . implode(', ', $columnas_existentes);
                }
            } else {
                $response['message'] = "ℹ️ Todas las columnas ya existían en la tabla: " . implode(', ', $columnas_existentes);
            }
        } else {
            $response['message'] = "Se encontraron errores al crear las columnas.";
            $response['details']['errores'] = $errores;
        }

    } catch (PDOException $e) {
        $response['message'] = "Error de base de datos: " . $e->getMessage();
        error_log("Error en generar_variables_respuesta_larga.php: " . $e->getMessage());
    } catch (Exception $e) {
        $response['message'] = "Error inesperado: " . $e->getMessage();
        error_log("Error inesperado en generar_variables_respuesta_larga.php: " . $e->getMessage());
    }
} else {
    $response['message'] = 'Método no permitido.';
}

echo json_encode($response);
?>
