<?php
session_start();

// Incluir configuración de cursos
require_once '../configuracion_cursos.php';

// Verificar que se recibieron los datos necesarios
if (!isset($_POST['curso']) || !isset($_POST['unidad']) || !isset($_POST['actividad']) || !isset($_POST['secciones'])) {
    if (isset($_POST['solo_vista_previa'])) {
        echo '<div style="color: #dc2626;">Error: Faltan parámetros necesarios</div>';
    } else {
        header('Content-Type: application/json');
        echo json_encode(['success' => false, 'error' => 'Faltan parámetros necesarios']);
    }
    exit;
}

$curso = $_POST['curso'];
$unidad = $_POST['unidad'];
$actividad = $_POST['actividad'];
$secciones = json_decode($_POST['secciones'], true);
$solo_servidor = isset($_POST['solo_servidor']) && $_POST['solo_servidor'];
$solo_vista_previa = isset($_POST['solo_vista_previa']) && $_POST['solo_vista_previa'];

// Validar datos
if (empty($curso) || empty($unidad) || empty($actividad) || !is_array($secciones)) {
    if ($solo_vista_previa) {
        echo '<div style="color: #dc2626;">Error: Datos inválidos</div>';
    } else {
        header('Content-Type: application/json');
        echo json_encode(['success' => false, 'error' => 'Datos inválidos']);
    }
    exit;
}

// Generar nombre del archivo
$numero_unidad = '';
if (preg_match('/Unidad (\d+)/', $unidad, $matches)) {
    $numero_unidad = sprintf('%02d', $matches[1]);
}

$numero_actividad = sprintf('%02d', $actividad);
$nombre_archivo = "{$curso}_u{$numero_unidad}_act{$numero_actividad}.php";

// Generar contenido del archivo PHP
$contenido = generarContenidoActividad($curso, $unidad, $actividad, $secciones);

// Si es solo vista previa, guardar archivo temporalmente y devolver el contenido HTML
if ($solo_vista_previa) {
    // Guardar el archivo temporalmente para que curso.php pueda encontrarlo
    $ruta_archivo = __DIR__ . '/../recursos/' . $nombre_archivo;
    if (file_put_contents($ruta_archivo, $contenido) === false) {
        echo '<div style="color: #dc2626;">Error: No se pudo crear el archivo temporal para vista previa</div>';
        exit;
    }
    
    // Extraer HTML y JavaScript del contenido para la vista previa
    $html_content = '';
    $lines = explode("\n", $contenido);
    $in_html = false;
    $in_script = false;
    
    foreach ($lines as $line) {
        // Detectar inicio de HTML
        if (strpos($line, '<link rel="stylesheet"') !== false || strpos($line, '<section') !== false) {
            $in_html = true;
        }
        
        // Detectar inicio de script
        if (strpos($line, '<script>') !== false) {
            $in_script = true;
        }
        
        // Detectar fin de script
        if (strpos($line, '</script>') !== false) {
            $in_script = false;
        }
        
        // Incluir HTML y JavaScript
        if ($in_html || $in_script) {
            $html_content .= $line . "\n";
        }
    }
    
    echo $html_content;
    exit;
}

// Si no es vista previa, proceder con el guardado normal
header('Content-Type: application/json');

// Guardar archivo en el servidor
$ruta_archivo = __DIR__ . '/../recursos/' . $nombre_archivo;
if (file_put_contents($ruta_archivo, $contenido) === false) {
    echo json_encode(['success' => false, 'error' => 'No se pudo crear el archivo en el servidor']);
    exit;
}

// Respuesta exitosa
echo json_encode([
    'success' => true, 
    'message' => 'Archivo guardado exitosamente en el servidor',
    'archivo' => $nombre_archivo,
    'secciones_guardadas' => count($secciones)
]);

function generarContenidoActividad($curso, $unidad, $actividad, $secciones) {
    $contenido = "<?php\n";
    $contenido .= "// Información de la actividad\n";
    $contenido .= "\$curso = '$curso';\n";
    $contenido .= "\$unidad = '$unidad';\n";
    $contenido .= "\$actividad = '$actividad';\n\n";
    $contenido .= "?>\n\n";
    $contenido .= '<link rel="stylesheet" href="recursos/actividades.css">' . "\n\n";

    // Generar secciones según el orden
    foreach ($secciones as $index => $seccion) {
        $contenido .= generarSeccionHTML($seccion, $index + 1, $curso, $unidad, $actividad);
    }

    return $contenido;
}


function generarSeccionHTML($seccion, $numero, $curso = '', $unidad = '', $actividad = '') {
    $html = "<!-- Sección " . $numero . ": " . $seccion['nombre'] . " -->\n";
    $html .= "<section class=\"" . $seccion['tipo'] . "\">\n";
    
    // Contenido específico según el tipo de sección
    switch ($seccion['tipo']) {
        case 'titulo-1':
            $descripcion = isset($seccion['contenido']['descripcion']) ? $seccion['contenido']['descripcion'] : '';
            // Usar el número de actividad desde la URL, eliminando ceros iniciales
            global $actividad;
            $numeroActividad = $actividad ? intval($actividad) : intval($numero);
            $html .= "    <h1 class=\"titulo-1\">Actividad " . $numeroActividad . ($descripcion ? ": " . htmlspecialchars($descripcion) : "") . "</h1>\n";
            break;
            
        case 'aprendizaje-esperado':
            $html .= "    <h3>Aprendizaje Esperado</h3>\n";
            
            // Si la sección tiene contenido, usarlo
            if (isset($seccion['contenido']['contenido'])) {
                $aprendizaje = $seccion['contenido']['contenido'];
            } else {
                // Si no tiene contenido, obtenerlo desde la configuración
                global $curso, $unidad;
                $contenido_config = obtenerAprendizajeEsperadoConfiguracion($curso, $unidad);
                $aprendizaje = $contenido_config['contenido'];
            }
            
            $html .= "    <p>" . htmlspecialchars($aprendizaje) . "</p>\n";
            break;
            
        case 'criterios-evaluacion':
            $html .= "    <h3>Criterios de Evaluación</h3>\n";
            $html .= "    \n";
            
            // Si la sección tiene contenido, usarlo
            if (isset($seccion['contenido']['criterios']) && !empty($seccion['contenido']['criterios'])) {
                $criterios = $seccion['contenido']['criterios'];
            } else {
                // Si no tiene contenido, obtenerlo desde la configuración
                global $curso, $unidad;
                $contenido_config = obtenerCriteriosEvaluacionConfiguracion($curso, $unidad);
                $criterios = $contenido_config['criterios'];
            }
            
            foreach ($criterios as $criterio) {
                $html .= "    <div class=\"criterio-item\">\n";
                $html .= "        <div class=\"criterio-codigo\">Criterio " . htmlspecialchars($criterio['codigo'] ?? '1.1.1') . "</div>\n";
                $html .= "        <div class=\"criterio-descripcion\">" . htmlspecialchars($criterio['descripcion'] ?? 'Descripción del criterio') . "</div>\n";
                $html .= "    </div>\n";
                $html .= "    \n";
            }
            break;
            
        case 'contenidos':
            $html .= "    <h3>Contenidos</h3>\n";
            $html .= "    \n";
            
            // Si la sección tiene contenido, usarlo
            if (isset($seccion['contenido']['contenidos']) && !empty($seccion['contenido']['contenidos'])) {
                $contenidos = $seccion['contenido']['contenidos'];
            } else {
                // Si no tiene contenido, obtenerlo desde la configuración
                global $curso, $unidad;
                $contenido_config = obtenerContenidosConfiguracion($curso, $unidad);
                $contenidos = $contenido_config['contenidos'];
            }
            
            foreach ($contenidos as $contenido) {
                $html .= "    <div class=\"contenido-item\">\n";
                $html .= "        <div class=\"contenido-header\">\n";
                $html .= "            <div class=\"contenido-icono\">\n";
                $html .= "                <i class=\"" . htmlspecialchars($contenido['icono'] ?? 'fas fa-book') . "\"></i>\n";
                $html .= "            </div>\n";
                $html .= "            <h4 class=\"contenido-titulo\">" . htmlspecialchars($contenido['titulo'] ?? 'Contenido') . "</h4>\n";
                $html .= "        </div>\n";
                $html .= "        <p class=\"contenido-descripcion\">" . htmlspecialchars($contenido['descripcion'] ?? 'Descripción del contenido') . "</p>\n";
                $html .= "    </div>\n";
                $html .= "    \n";
            }
            break;
            
        case 'archivos-pdf':
            $titulo = isset($seccion['contenido']['titulo']) ? $seccion['contenido']['titulo'] : 'Guía de la Actividad';
            $descripcion = isset($seccion['contenido']['descripcion']) ? $seccion['contenido']['descripcion'] : 'Documento con las instrucciones y material de apoyo para la actividad.';
            $archivo = isset($seccion['contenido']['archivo']) ? $seccion['contenido']['archivo'] : null;
            
            $html .= "    <h3>Documento PDF</h3>\n";
            $html .= "    \n";
            $html .= "    <div class=\"pdf-item\">\n";
            $html .= "        <div class=\"pdf-header\">\n";
            $html .= "            <div class=\"pdf-icono\">\n";
            $html .= "                <i class=\"fas fa-file-pdf\"></i>\n";
            $html .= "            </div>\n";
            $html .= "            <h4 class=\"pdf-titulo\">" . htmlspecialchars($titulo) . "</h4>\n";
            $html .= "        </div>\n";
            $html .= "        <p class=\"pdf-descripcion\">" . htmlspecialchars($descripcion) . "</p>\n";
            
            if ($archivo) {
                $html .= "        \n";
                $html .= "        <div class=\"pdf-preview\" style=\"min-height: 600px; display: block; text-align: left; padding: 0; border: none; background: white; margin: 16px 0;\">\n";
                $html .= "            <iframe \n";
                $html .= "                src=\"documentos/" . htmlspecialchars($archivo) . "#toolbar=1&navpanes=1&scrollbar=1\" \n";
                $html .= "                width=\"100%\" \n";
                $html .= "                height=\"600\" \n";
                $html .= "                style=\"border: 1px solid #ddd; border-radius: 8px; box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);\" \n";
                $html .= "                title=\"" . htmlspecialchars($titulo) . "\">\n";
                $html .= "            </iframe>\n";
                $html .= "        </div>\n";
                $html .= "        \n";
                $html .= "        <div class=\"pdf-botones\">\n";
                $html .= "            <button class=\"pdf-abrir\" onclick=\"window.open('documentos/" . htmlspecialchars($archivo) . "', '_blank')\">\n";
                $html .= "                <i class=\"fas fa-external-link-alt\"></i>\n";
                $html .= "                Abrir en Nueva Pestaña\n";
                $html .= "            </button>\n";
            $html .= "            <a href=\"documentos/" . htmlspecialchars($archivo) . "\" download class=\"pdf-descargar\" style=\"text-decoration: none; display: inline-flex; align-items: center; justify-content: center;\">\n";
            $html .= "                <i class=\"fas fa-download\"></i>\n";
            $html .= "                Descargar PDF\n";
            $html .= "            </a>\n";
                $html .= "        </div>\n";
            }
            
            $html .= "    </div>\n";
            break;
            
        case 'seccion-video':
            $titulo = isset($seccion['contenido']['titulo']) ? $seccion['contenido']['titulo'] : 'Tutorial de la Actividad';
            $descripcion = isset($seccion['contenido']['descripcion']) ? $seccion['contenido']['descripcion'] : 'Video explicativo con los pasos para completar la actividad correctamente.';
            $duracion = isset($seccion['contenido']['duracion']) ? $seccion['contenido']['duracion'] : '';
            $url = isset($seccion['contenido']['url']) ? $seccion['contenido']['url'] : '';
            $archivo = isset($seccion['contenido']['archivo']) ? $seccion['contenido']['archivo'] : '';
            $tipo_contenido = isset($seccion['contenido']['tipo_contenido']) ? $seccion['contenido']['tipo_contenido'] : 'url';
            
            $html .= "    <h3>Video Educativo</h3>\n";
            $html .= "    \n";
            $html .= "    <div class=\"video-container\">\n";
            $html .= "        <div class=\"video-titulo\">" . htmlspecialchars($titulo) . "</div>\n";
            $html .= "        <div class=\"video-descripcion\">\n";
            $html .= "            " . htmlspecialchars($descripcion) . "\n";
            $html .= "        </div>\n";
            
            if ($tipo_contenido === 'url' && $url) {
                // Convertir URL de YouTube a embed si es necesario
                $url_embed = $url;
                if (strpos($url, 'youtube.com/watch') !== false) {
                    $url_embed = str_replace('youtube.com/watch?v=', 'youtube.com/embed/', $url);
                } elseif (strpos($url, 'youtu.be/') !== false) {
                    $url_embed = str_replace('youtu.be/', 'youtube.com/embed/', $url);
                } elseif (strpos($url, 'youtube.com/live/') !== false) {
                    $url_embed = str_replace('youtube.com/live/', 'youtube.com/embed/', $url);
                }
                
                $html .= "        <div class=\"video-embed\">\n";
                $html .= "            <iframe \n";
                $html .= "                src=\"" . htmlspecialchars($url_embed) . "\" \n";
                $html .= "                title=\"" . htmlspecialchars($titulo) . "\"\n";
                $html .= "                allow=\"accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture\" \n";
                $html .= "                allowfullscreen>\n";
                $html .= "            </iframe>\n";
                $html .= "            <div style=\"margin-top: 1rem; text-align: center;\">\n";
                $html .= "                <a href=\"" . htmlspecialchars($url) . "\" target=\"_blank\" class=\"btn-pdf\" style=\"background: #007bff; color: white; padding: 0.5rem 1rem; border-radius: 4px; text-decoration: none; display: inline-block;\">\n";
                $html .= "                    <i class=\"fas fa-external-link-alt\"></i> Abrir en YouTube\n";
                $html .= "                </a>\n";
                $html .= "            </div>\n";
                $html .= "        </div>\n";
                $plataforma = 'YouTube';
            } elseif ($archivo) {
                $html .= "        <div class=\"video-embed\">\n";
                $html .= "            <video \n";
                $html .= "                src=\"videos/" . htmlspecialchars($archivo) . "\" \n";
                $html .= "                controls \n";
                $html .= "                style=\"width: 100%; height: 400px;\"\n";
                $html .= "                title=\"" . htmlspecialchars($titulo) . "\">\n";
                $html .= "            </video>\n";
                $html .= "        </div>\n";
                $plataforma = 'Archivo Local';
            } else {
                $html .= "        <div class=\"video-embed\">\n";
                $html .= "            <div style=\"padding: 40px; text-align: center; background: #f8f9fa; border: 2px dashed #dee2e6; border-radius: 8px;\">\n";
                $html .= "                <i class=\"fas fa-video\" style=\"font-size: 3rem; color: #6c757d; margin-bottom: 1rem;\"></i>\n";
                $html .= "                <p style=\"color: #6c757d; margin: 0;\">No hay video configurado</p>\n";
                $html .= "            </div>\n";
                $html .= "        </div>\n";
                $plataforma = 'Sin contenido';
            }
            
            $html .= "        <div class=\"video-info\">\n";
            if ($duracion) {
                $html .= "            <span class=\"video-duracion\">⏱️ " . htmlspecialchars($duracion) . "</span>\n";
            }
            $html .= "            <span class=\"video-plataforma\">" . htmlspecialchars($plataforma) . "</span>\n";
            $html .= "        </div>\n";
            $html .= "    </div>\n";
            break;
            
        case 'seccion-podcast':
            $titulo = isset($seccion['contenido']['titulo']) ? $seccion['contenido']['titulo'] : 'Audio Explicativo';
            $descripcion = isset($seccion['contenido']['descripcion']) ? $seccion['contenido']['descripcion'] : 'Audio con explicaciones detalladas sobre los conceptos de la actividad.';
            $duracion = isset($seccion['contenido']['duracion']) ? $seccion['contenido']['duracion'] : '';
            $archivo = isset($seccion['contenido']['archivo']) ? $seccion['contenido']['archivo'] : '';
            
            $html .= "    <h3>Podcast Educativo</h3>\n";
            $html .= "    \n";
            $html .= "    <div class=\"podcast-container\">\n";
            $html .= "        <div class=\"podcast-titulo\">" . htmlspecialchars($titulo) . "</div>\n";
            $html .= "        <div class=\"podcast-descripcion\">\n";
            $html .= "            " . htmlspecialchars($descripcion) . "\n";
            $html .= "        </div>\n";
            
            if ($archivo) {
                $html .= "        \n";
                $html .= "        <div class=\"podcast-player\">\n";
                $html .= "            <audio \n";
                $html .= "                src=\"podcasts/" . htmlspecialchars($archivo) . "\" \n";
                $html .= "                controls \n";
                $html .= "                style=\"width: 100%; margin: 1rem 0;\">\n";
                $html .= "                Tu navegador no soporta el elemento audio.\n";
                $html .= "            </audio>\n";
                $html .= "        </div>\n";
                $plataforma = 'Audio MP3';
            } else {
                $html .= "        \n";
                $html .= "        <div class=\"podcast-player\">\n";
                $html .= "            <div style=\"padding: 40px; text-align: center; background: #f8f9fa; border: 2px dashed #dee2e6; border-radius: 8px;\">\n";
                $html .= "                <i class=\"fas fa-headphones\" style=\"font-size: 3rem; color: #6c757d; margin-bottom: 1rem;\"></i>\n";
                $html .= "                <p style=\"color: #6c757d; margin: 0;\">No hay audio configurado</p>\n";
                $html .= "            </div>\n";
                $html .= "        </div>\n";
                $plataforma = 'Sin contenido';
            }
            
            $html .= "        \n";
            $html .= "        <div class=\"podcast-info\">\n";
            if ($duracion) {
                $html .= "            <span class=\"podcast-duracion\">⏱️ " . htmlspecialchars($duracion) . "</span>\n";
            }
            $html .= "            <span class=\"podcast-plataforma\">" . htmlspecialchars($plataforma) . "</span>\n";
            $html .= "        </div>\n";
            $html .= "    </div>\n";
            break;
            
        case 'respuesta-larga':
            $titulo_principal = isset($seccion['contenido']['titulo_principal']) ? $seccion['contenido']['titulo_principal'] : 'Respuesta Larga 01';
            $subtitulo = isset($seccion['contenido']['subtitulo']) ? $seccion['contenido']['subtitulo'] : 'Respuesta del Estudiante';
            $descripcion = isset($seccion['contenido']['descripcion']) ? $seccion['contenido']['descripcion'] : 'Escribe tu respuesta completa a la actividad. Asegúrate de incluir todos los puntos solicitados y fundamentar tus respuestas con ejemplos cuando sea necesario.';
            $min_palabras = isset($seccion['contenido']['min_palabras']) ? $seccion['contenido']['min_palabras'] : 50;
            $max_palabras = isset($seccion['contenido']['max_palabras']) ? $seccion['contenido']['max_palabras'] : 100;
            $incluir_fecha_limite = isset($seccion['contenido']['incluir_fecha_limite']) ? $seccion['contenido']['incluir_fecha_limite'] : false;
            $fecha_limite = isset($seccion['contenido']['fecha_limite']) ? $seccion['contenido']['fecha_limite'] : '';
            $incluir_retroalimentacion = isset($seccion['contenido']['incluir_retroalimentacion']) ? $seccion['contenido']['incluir_retroalimentacion'] : false;
            $retro_descripcion = isset($seccion['contenido']['retro_descripcion']) ? $seccion['contenido']['retro_descripcion'] : 'Proporciona retroalimentación constructiva sobre la respuesta del estudiante. Incluye comentarios sobre fortalezas, áreas de mejora y sugerencias para el aprendizaje.';
            $incluir_calificacion = isset($seccion['contenido']['incluir_calificacion']) ? $seccion['contenido']['incluir_calificacion'] : false;
            
            $html .= "    <h3>" . htmlspecialchars($titulo_principal) . "</h3>\n";
            $html .= "    \n";
            $html .= "    <div class=\"espacio-respuesta\">\n";
            $html .= "        <h4>" . htmlspecialchars($subtitulo) . "</h4>\n";
            $html .= "        <div class=\"espacio-respuesta-descripcion\">\n";
            $html .= "            " . htmlspecialchars($descripcion) . "\n";
            $html .= "        </div>\n";
            // Obtener el número de respuesta larga para los IDs
            $numero_respuesta = isset($seccion['numero_respuesta']) ? $seccion['numero_respuesta'] : 1;
            $numero_formateado = str_pad($numero_respuesta, 2, '0', STR_PAD_LEFT);
            
            $html .= "        <textarea class=\"textarea-respuesta\" id=\"respuesta-estudiante-{$numero_formateado}\" placeholder=\"Escribe tu respuesta aquí (mínimo {$min_palabras} palabras, máximo {$max_palabras} palabras)...\"></textarea>\n";
            $html .= "        <div class=\"mensaje-exito-respuesta\" id=\"mensaje-exito-respuesta-{$numero_formateado}\" style=\"display: none;\">\n";
            $html .= "            <i class=\"fas fa-check-circle\"></i>\n";
            $html .= "            <span>Respuesta enviada con éxito</span>\n";
            $html .= "        </div>\n";
            $html .= "        <div class=\"contador-palabras\" id=\"contador-palabras-{$numero_formateado}\">0 palabras</div>\n";
            $html .= "        <div class=\"fecha-actualizacion\" id=\"fecha-respuesta-{$numero_formateado}\" style=\"display: none;\"></div>\n";
            $html .= "        <div class=\"fecha-limite-container\" id=\"fecha-limite-{$numero_formateado}\" style=\"display: none;\">\n";
            $html .= "            <div class=\"fecha-limite-label\">📅 Fecha Límite:</div>\n";
            $html .= "            <div class=\"fecha-limite-value\" id=\"fecha-limite-value-{$numero_formateado}\"></div>\n";
            $html .= "        </div>\n";
            $html .= "        <div class=\"botones-guardar\">\n";
            $html .= "            <button class=\"btn-guardar\" onclick=\"guardarRespuestaLarga({$numero_respuesta}, '{$curso}', '{$unidad}', '{$actividad}')\" id=\"btn-enviar-respuesta-{$numero_formateado}\">\n";
            $html .= "                <i class=\"fas fa-save\"></i>\n";
            $html .= "                <span class=\"btn-text\">Enviar Respuesta</span>\n";
            $html .= "            </button>\n";
            $html .= "            <button class=\"btn-eliminar-respuesta\" onclick=\"eliminarRespuestaLarga({$numero_respuesta}, '{$curso}', '{$unidad}', '{$actividad}')\" style=\"display: none;\" id=\"btn-eliminar-respuesta-{$numero_formateado}\">\n";
            $html .= "                <i class=\"fas fa-trash\"></i>\n";
            $html .= "                Eliminar Respuesta\n";
            $html .= "            </button>\n";
            $html .= "        </div>\n";
            
            if ($incluir_retroalimentacion) {
                $html .= "        \n";
                $html .= "        <div class=\"retroalimentacion-mejorada\">\n";
                $html .= "            <h4>Retroalimentación del Profesor</h4>\n";
                $html .= "            <div class=\"espacio-respuesta-descripcion\">\n";
                $html .= "                " . htmlspecialchars($retro_descripcion) . "\n";
                $html .= "            </div>\n";
                $html .= "            <textarea class=\"textarea-retro-mejorada\" id=\"retro-respuesta-larga-{$numero_formateado}\" placeholder=\"Escribe aquí la retroalimentación para el estudiante...\"></textarea>\n";
                $html .= "            <div class=\"fecha-actualizacion\" id=\"fecha-retroalimentacion-{$numero_formateado}\" style=\"display: none;\"></div>\n";
                $html .= "            \n";
                
                if ($incluir_calificacion) {
                    $html .= "            <div class=\"calificacion-retro-mejorada-container\">\n";
                    $html .= "                <span class=\"calificacion-retro-mejorada-label\">Calificar:</span>\n";
                    $html .= "                <input type=\"number\" class=\"calificacion-retro-mejorada-input\" id=\"calificacion-respuesta-larga-{$numero_formateado}\" min=\"0\" max=\"100\" placeholder=\"0\">\n";
                    $html .= "                <span class=\"calificacion-retro-mejorada-maximo\">/ 100</span>\n";
                    $html .= "                <button class=\"btn-guardar-retro-mejorada\" onclick=\"guardarNotaRespuestaLarga({$numero_respuesta}, '{$curso}', '{$unidad}', '{$actividad}')\">\n";
                    $html .= "                    <i class=\"fas fa-star\"></i>\n";
                    $html .= "                    Guardar Nota\n";
                    $html .= "                </button>\n";
                    $html .= "            </div>\n";
                    $html .= "            <div class=\"fecha-actualizacion\" id=\"fecha-nota-{$numero_formateado}\" style=\"display: none;\"></div>\n";
                }
                
                $html .= "            \n";
                $html .= "            <button class=\"btn-guardar-comentario-retro-mejorada\" onclick=\"guardarRetroalimentacionRespuestaLarga({$numero_respuesta}, '{$curso}', '{$unidad}', '{$actividad}')\">\n";
                $html .= "                <i class=\"fas fa-comment\"></i>\n";
                $html .= "                Guardar Retroalimentación\n";
                $html .= "            </button>\n";
                $html .= "            <button class=\"btn-eliminar-respuesta\" onclick=\"eliminarRetroalimentacionRespuestaLarga({$numero_respuesta}, '{$curso}', '{$unidad}', '{$actividad}')\" style=\"display: none;\" id=\"btn-eliminar-retroalimentacion-{$numero_formateado}\">\n";
                $html .= "                <i class=\"fas fa-trash\"></i>\n";
                $html .= "                Eliminar Retroalimentación\n";
                $html .= "            </button>\n";
                $html .= "        </div>\n";
            }
            
            $html .= "    </div>\n";
            
            // Agregar JavaScript para el contador de palabras
            $html .= "<script>\n";
            $html .= "document.addEventListener('DOMContentLoaded', function() {\n";
            $html .= "    const textarea = document.getElementById('respuesta-estudiante-{$numero_formateado}');\n";
            $html .= "    const contador = document.getElementById('contador-palabras-{$numero_formateado}');\n";
            $html .= "    const btnGuardar = document.querySelector('.btn-guardar');\n";
            $html .= "    \n";
            $html .= "    if (textarea && contador) {\n";
            $html .= "        console.log('📝 Contador de palabras inicializado para Respuesta Larga');\n";
            $html .= "        \n";
            $html .= "        // Función para contar palabras\n";
            $html .= "        function contarPalabras(texto) {\n";
            $html .= "            const textoLimpio = texto.trim();\n";
            $html .= "            if (textoLimpio === '') return 0;\n";
            $html .= "            return textoLimpio.split(/\\s+/).length;\n";
            $html .= "        }\n";
            $html .= "        \n";
            $html .= "        // Función para validar el rango de palabras\n";
            $html .= "        function validarRangoPalabras(palabras) {\n";
            $html .= "            const minPalabras = {$min_palabras};\n";
            $html .= "            const maxPalabras = {$max_palabras};\n";
            $html .= "            return palabras >= minPalabras && palabras <= maxPalabras;\n";
            $html .= "        }\n";
            $html .= "        \n";
            $html .= "        // Función para actualizar el contador y validar el botón\n";
            $html .= "        function actualizarContador() {\n";
            $html .= "            const texto = textarea.value;\n";
            $html .= "            const palabras = contarPalabras(texto);\n";
            $html .= "            const minPalabras = {$min_palabras};\n";
            $html .= "            const maxPalabras = {$max_palabras};\n";
            $html .= "            \n";
            $html .= "            console.log('📊 Palabras contadas:', palabras);\n";
            $html .= "            contador.textContent = palabras + ' palabras';\n";
            $html .= "            \n";
            $html .= "            // Cambiar color según el rango de palabras\n";
            $html .= "            if (palabras < minPalabras) {\n";
            $html .= "                contador.style.color = '#ef4444'; // Rojo - insuficiente\n";
            $html .= "                contador.style.fontWeight = 'bold';\n";
            $html .= "            } else if (palabras > maxPalabras) {\n";
            $html .= "                contador.style.color = '#ef4444'; // Rojo - excesivo\n";
            $html .= "                contador.style.fontWeight = 'bold';\n";
            $html .= "            } else {\n";
            $html .= "                contador.style.color = '#10b981'; // Verde - rango ideal\n";
            $html .= "                contador.style.fontWeight = 'bold';\n";
            $html .= "            }\n";
            $html .= "            \n";
            $html .= "            // Validar y actualizar el botón de guardar\n";
            $html .= "            if (btnGuardar) {\n";
            $html .= "                const enRangoIdeal = validarRangoPalabras(palabras);\n";
            $html .= "                \n";
            $html .= "                if (enRangoIdeal) {\n";
            $html .= "                    btnGuardar.disabled = false;\n";
            $html .= "                    btnGuardar.style.opacity = '1';\n";
            $html .= "                    btnGuardar.style.cursor = 'pointer';\n";
            $html .= "                    console.log('✅ Botón habilitado - rango ideal');\n";
            $html .= "                } else {\n";
            $html .= "                    btnGuardar.disabled = true;\n";
            $html .= "                    btnGuardar.style.opacity = '0.6';\n";
            $html .= "                    btnGuardar.style.cursor = 'not-allowed';\n";
            $html .= "                    console.log('❌ Botón deshabilitado - fuera del rango ideal');\n";
            $html .= "                }\n";
            $html .= "            }\n";
            $html .= "        }\n";
            $html .= "        \n";
            $html .= "        // Event listener para el textarea\n";
            $html .= "        textarea.addEventListener('input', actualizarContador);\n";
            $html .= "        \n";
            $html .= "        // Inicializar contador y validación\n";
            $html .= "        actualizarContador();\n";
            $html .= "        \n";
            $html .= "        // El botón \"Guardar Respuesta\" usa la función onclick=\"guardarRespuestaLarga()\" \n";
            $html .= "        // que está definida más abajo en el script\n";
            $html .= "        console.log('✅ Botón \"Guardar Respuesta\" configurado con onclick');\n";
            $html .= "        \n";
            $html .= "        // Los botones de retroalimentación y nota usan onclick=\"guardarRetroalimentacionRespuestaLarga()\" \n";
            $html .= "        // y onclick=\"guardarNotaRespuestaLarga()\" que están definidas más abajo en el script\n";
            $html .= "        console.log('✅ Botones de retroalimentación y nota configurados con onclick');\n";
            $html .= "    }\n";
            $html .= "    \n";
            $html .= "    // Cargar datos existentes al final para asegurar que todo esté completamente renderizado\n";
            $html .= "    setTimeout(() => {\n";
            $html .= "        console.log('🔄 Iniciando carga de datos después de timeout...');\n";
            $html .= "        cargarDatosExistentes();\n";
            $html .= "    }, 100);\n";
            $html .= "});\n";
            $html .= "\n";
            $html .= "// Funciones globales para guardar en base de datos\n";
            $html .= "async function guardarRespuestaLarga(numeroRespuesta, curso, unidad, actividad) {\n";
            $html .= "    console.log('🚀 Iniciando guardado de respuesta...');\n";
            $html .= "    console.log('📋 Parámetros:', { numeroRespuesta, curso, unidad, actividad });\n";
            $html .= "    \n";
            $html .= "    const numeroFormateado = numeroRespuesta.toString().padStart(2, '0');\n";
            $html .= "    const textarea = document.getElementById('respuesta-estudiante-' + numeroFormateado);\n";
            $html .= "    \n";
            $html .= "    if (!textarea) {\n";
            $html .= "        alert('❌ Error: No se encontró el área de texto para la respuesta');\n";
            $html .= "        console.error('❌ Textarea no encontrado:', 'respuesta-estudiante-' + numeroFormateado);\n";
            $html .= "        return;\n";
            $html .= "    }\n";
            $html .= "    \n";
            $html .= "    const respuesta = textarea.value.trim();\n";
            $html .= "    const palabras = respuesta.split(/\\s+/).filter(palabra => palabra.length > 0).length;\n";
            $html .= "    console.log('📝 Respuesta a guardar:', respuesta);\n";
            $html .= "    console.log('📊 Palabras contadas:', palabras);\n";
            $html .= "    \n";
            $html .= "    if (!respuesta) {\n";
            $html .= "        alert('Por favor, escribe una respuesta antes de guardar.');\n";
            $html .= "        return;\n";
            $html .= "    }\n";
            $html .= "    \n";
            $html .= "    const datosEnvio = {\n";
            $html .= "        tipo: 'respuesta',\n";
            $html .= "        numero_respuesta: numeroRespuesta,\n";
            $html .= "        curso: curso,\n";
            $html .= "        unidad: unidad,\n";
            $html .= "        actividad: actividad,\n";
            $html .= "        contenido: respuesta,\n";
            $html .= "        palabras: palabras\n";
            $html .= "    };\n";
            $html .= "    \n";
            $html .= "    console.log('📤 Datos a enviar:', datosEnvio);\n";
            $html .= "    \n";
            $html .= "    try {\n";
            $html .= "        console.log('🌐 Enviando petición a: api/guardar_respuesta_larga.php');\n";
            $html .= "        \n";
            $html .= "        const response = await fetch('api/guardar_respuesta_larga.php', {\n";
            $html .= "            method: 'POST',\n";
            $html .= "            headers: { 'Content-Type': 'application/json' },\n";
            $html .= "            body: JSON.stringify(datosEnvio)\n";
            $html .= "        });\n";
            $html .= "        \n";
            $html .= "        console.log('📡 Respuesta recibida:', response);\n";
            $html .= "        console.log('📊 Status:', response.status);\n";
            $html .= "        \n";
            $html .= "        if (!response.ok) {\n";
            $html .= "            throw new Error(`HTTP error! status: \${response.status}`);\n";
            $html .= "        }\n";
            $html .= "        \n";
            $html .= "        const result = await response.json();\n";
            $html .= "        console.log('📋 Resultado:', result);\n";
            $html .= "        \n";
            $html .= "        if (result.success) {\n";
            $html .= "            // Aplicar estilo de guardado\n";
            $html .= "            textarea.classList.add('guardado');\n";
            $html .= "            \n";
            $html .= "            // Actualizar texto del botón a 'Respuesta Enviada' y deshabilitarlo\n";
            $html .= "            const btnEnviar = document.getElementById('btn-enviar-respuesta-' + numeroFormateado);\n";
            $html .= "            if (btnEnviar) {\n";
            $html .= "                const btnText = btnEnviar.querySelector('.btn-text');\n";
            $html .= "                if (btnText) {\n";
            $html .= "                    btnText.textContent = 'Respuesta Enviada';\n";
            $html .= "                }\n";
            $html .= "                btnEnviar.disabled = true;\n";
            $html .= "            }\n";
            $html .= "            \n";
            $html .= "            // Mostrar mensaje de éxito permanentemente\n";
            $html .= "            const mensajeExito = document.getElementById('mensaje-exito-respuesta-' + numeroFormateado);\n";
            $html .= "            if (mensajeExito) {\n";
            $html .= "                mensajeExito.style.display = 'flex';\n";
            $html .= "            }\n";
            $html .= "            \n";
            $html .= "            // Mostrar fecha de actualización\n";
            $html .= "            const fechaElement = document.getElementById('fecha-respuesta-' + numeroFormateado);\n";
            $html .= "            if (fechaElement) {\n";
            $html .= "                fechaElement.textContent = 'Última actualización: ' + new Date().toLocaleString();\n";
            $html .= "                fechaElement.style.display = 'block';\n";
            $html .= "            }\n";
            $html .= "            \n";
            $html .= "            // Mostrar botón de eliminar\n";
            $html .= "            const btnEliminar = document.getElementById('btn-eliminar-respuesta-' + numeroFormateado);\n";
            $html .= "            if (btnEliminar) {\n";
            $html .= "                btnEliminar.style.display = 'inline-block';\n";
            $html .= "            }\n";
            $html .= "            \n";
            $html .= "            // Recargar página para actualizar el estado del botón de término de actividad\n";
            $html .= "            setTimeout(() => {\n";
            $html .= "                window.location.reload();\n";
            $html .= "            }, 100);\n";
            $html .= "        } else {\n";
            $html .= "            alert('❌ Error al guardar: ' + (result.message || 'Error desconocido'));\n";
            $html .= "        }\n";
            $html .= "    } catch (error) {\n";
            $html .= "        console.error('❌ Error completo:', error);\n";
            $html .= "        alert('❌ Error de conexión: ' + error.message);\n";
            $html .= "    }\n";
            $html .= "}\n";
            $html .= "\n";
            $html .= "async function guardarRetroalimentacionRespuestaLarga(numeroRespuesta, curso, unidad, actividad) {\n";
            $html .= "    console.log('🚀 Iniciando guardado de retroalimentación...');\n";
            $html .= "    \n";
            $html .= "    const numeroFormateado = numeroRespuesta.toString().padStart(2, '0');\n";
            $html .= "    const textarea = document.getElementById('retro-respuesta-larga-' + numeroFormateado);\n";
            $html .= "    \n";
            $html .= "    if (!textarea) {\n";
            $html .= "        alert('❌ Error: No se encontró el área de texto para la retroalimentación');\n";
            $html .= "        console.error('❌ Textarea no encontrado:', 'retro-respuesta-larga-' + numeroFormateado);\n";
            $html .= "        return;\n";
            $html .= "    }\n";
            $html .= "    \n";
            $html .= "    const retroalimentacion = textarea.value.trim();\n";
            $html .= "    console.log('📝 Retroalimentación a guardar:', retroalimentacion);\n";
            $html .= "    \n";
            $html .= "    if (!retroalimentacion) {\n";
            $html .= "        alert('Por favor, escribe una retroalimentación antes de guardar.');\n";
            $html .= "        return;\n";
            $html .= "    }\n";
            $html .= "    \n";
            $html .= "    try {\n";
            $html .= "        const response = await fetch('api/guardar_respuesta_larga.php', {\n";
            $html .= "            method: 'POST',\n";
            $html .= "            headers: { 'Content-Type': 'application/json' },\n";
            $html .= "            body: JSON.stringify({\n";
            $html .= "                tipo: 'retroalimentacion',\n";
            $html .= "                numero_respuesta: numeroRespuesta,\n";
            $html .= "                curso: curso,\n";
            $html .= "                unidad: unidad,\n";
            $html .= "                actividad: actividad,\n";
            $html .= "                contenido: retroalimentacion\n";
            $html .= "            })\n";
            $html .= "        });\n";
            $html .= "        \n";
            $html .= "        console.log('📡 Status de retroalimentación:', response.status);\n";
            $html .= "        const result = await response.json();\n";
            $html .= "        console.log('📋 Resultado retroalimentación:', result);\n";
            $html .= "        \n";
            $html .= "        if (result.success) {\n";
            $html .= "            alert('✅ Retroalimentación guardada exitosamente');\n";
            $html .= "            \n";
            $html .= "            // Aplicar estilo de guardado\n";
            $html .= "            textarea.classList.add('guardado');\n";
            $html .= "            \n";
            $html .= "            // Mostrar fecha de actualización\n";
            $html .= "            const fechaElement = document.getElementById('fecha-retroalimentacion-' + numeroFormateado);\n";
            $html .= "            if (fechaElement) {\n";
            $html .= "                fechaElement.textContent = 'Última actualización: ' + new Date().toLocaleString();\n";
            $html .= "                fechaElement.style.display = 'block';\n";
            $html .= "            }\n";
            $html .= "            \n";
            $html .= "            // Mostrar botón de eliminar\n";
            $html .= "            const btnEliminar = document.getElementById('btn-eliminar-retroalimentacion-' + numeroFormateado);\n";
            $html .= "            if (btnEliminar) {\n";
            $html .= "                btnEliminar.style.display = 'inline-block';\n";
            $html .= "            }\n";
            $html .= "        } else {\n";
            $html .= "            alert('❌ Error al guardar: ' + (result.message || 'Error desconocido'));\n";
            $html .= "        }\n";
            $html .= "    } catch (error) {\n";
            $html .= "        console.error('❌ Error en retroalimentación:', error);\n";
            $html .= "        alert('❌ Error de conexión: ' + error.message);\n";
            $html .= "    }\n";
            $html .= "}\n";
            $html .= "\n";
            $html .= "async function guardarNotaRespuestaLarga(numeroRespuesta, curso, unidad, actividad) {\n";
            $html .= "    console.log('🚀 Iniciando guardado de nota...');\n";
            $html .= "    \n";
            $html .= "    const numeroFormateado = numeroRespuesta.toString().padStart(2, '0');\n";
            $html .= "    const input = document.getElementById('calificacion-respuesta-larga-' + numeroFormateado);\n";
            $html .= "    \n";
            $html .= "    if (!input) {\n";
            $html .= "        alert('❌ Error: No se encontró el campo de calificación');\n";
            $html .= "        console.error('❌ Input no encontrado:', 'calificacion-respuesta-larga-' + numeroFormateado);\n";
            $html .= "        return;\n";
            $html .= "    }\n";
            $html .= "    \n";
            $html .= "    const nota = parseFloat(input.value);\n";
            $html .= "    console.log('📊 Nota a guardar:', nota);\n";
            $html .= "    \n";
            $html .= "    if (isNaN(nota) || nota < 0 || nota > 100) {\n";
            $html .= "        alert('Por favor, ingresa una nota válida entre 0 y 100.');\n";
            $html .= "        return;\n";
            $html .= "    }\n";
            $html .= "    \n";
            $html .= "    try {\n";
            $html .= "        const response = await fetch('api/guardar_respuesta_larga.php', {\n";
            $html .= "            method: 'POST',\n";
            $html .= "            headers: { 'Content-Type': 'application/json' },\n";
            $html .= "            body: JSON.stringify({\n";
            $html .= "                tipo: 'nota',\n";
            $html .= "                numero_respuesta: numeroRespuesta,\n";
            $html .= "                curso: curso,\n";
            $html .= "                unidad: unidad,\n";
            $html .= "                actividad: actividad,\n";
            $html .= "                contenido: nota\n";
            $html .= "            })\n";
            $html .= "        });\n";
            $html .= "        \n";
            $html .= "        console.log('📡 Status de nota:', response.status);\n";
            $html .= "        const result = await response.json();\n";
            $html .= "        console.log('📋 Resultado nota:', result);\n";
            $html .= "        \n";
            $html .= "        if (result.success) {\n";
            $html .= "            alert('✅ Nota guardada exitosamente');\n";
            $html .= "            \n";
            $html .= "            // Aplicar estilo de guardado\n";
            $html .= "            input.classList.add('guardado');\n";
            $html .= "            \n";
            $html .= "            // Mostrar fecha de actualización\n";
            $html .= "            const fechaElement = document.getElementById('fecha-nota-' + numeroFormateado);\n";
            $html .= "            if (fechaElement) {\n";
            $html .= "                fechaElement.textContent = 'Última actualización: ' + new Date().toLocaleString();\n";
            $html .= "                fechaElement.style.display = 'block';\n";
            $html .= "            }\n";
            $html .= "        } else {\n";
            $html .= "            alert('❌ Error al guardar: ' + (result.message || 'Error desconocido'));\n";
            $html .= "        }\n";
            $html .= "    } catch (error) {\n";
            $html .= "        console.error('❌ Error en nota:', error);\n";
            $html .= "        alert('❌ Error de conexión: ' + error.message);\n";
            $html .= "    }\n";
            $html .= "}\n";
            $html .= "\n";
            $html .= "// Funciones para eliminar respuestas y retroalimentación\n";
            $html .= "async function eliminarRespuestaLarga(numeroRespuesta, curso, unidad, actividad) {\n";
            $html .= "    if (!confirm('¿Estás seguro de que quieres eliminar completamente esta respuesta? Esta acción no se puede deshacer.')) {\n";
            $html .= "        return;\n";
            $html .= "    }\n";
            $html .= "    \n";
            $html .= "    const numeroFormateado = numeroRespuesta.toString().padStart(2, '0');\n";
            $html .= "    const textarea = document.getElementById('respuesta-estudiante-' + numeroFormateado);\n";
            $html .= "    \n";
            $html .= "    try {\n";
            $html .= "        const response = await fetch('api/guardar_respuesta_larga.php', {\n";
            $html .= "            method: 'POST',\n";
            $html .= "            headers: { 'Content-Type': 'application/json' },\n";
            $html .= "            body: JSON.stringify({\n";
            $html .= "                tipo: 'eliminar_respuesta',\n";
            $html .= "                numero_respuesta: numeroRespuesta,\n";
            $html .= "                curso: curso,\n";
            $html .= "                unidad: unidad,\n";
            $html .= "                actividad: actividad,\n";
            $html .= "                contenido: null\n";
            $html .= "            })\n";
            $html .= "        });\n";
            $html .= "        \n";
            $html .= "        const result = await response.json();\n";
            $html .= "        if (result.success) {\n";
            $html .= "            // Limpiar el campo y remover estilos\n";
            $html .= "            textarea.value = '';\n";
            $html .= "            textarea.classList.remove('guardado');\n";
            $html .= "            \n";
            $html .= "            // Actualizar contador manualmente\n";
            $html .= "            const contador = document.getElementById('contador-palabras-' + numeroFormateado);\n";
            $html .= "            if (contador) {\n";
            $html .= "                contador.textContent = '0 palabras';\n";
            $html .= "                contador.style.color = '#6b7280';\n";
            $html .= "                contador.style.fontWeight = 'normal';\n";
            $html .= "            }\n";
            $html .= "            \n";
            $html .= "            // Restaurar texto del botón a 'Enviar Respuesta' y habilitarlo\n";
            $html .= "            const btnEnviar = document.getElementById('btn-enviar-respuesta-' + numeroFormateado);\n";
            $html .= "            if (btnEnviar) {\n";
            $html .= "                const btnText = btnEnviar.querySelector('.btn-text');\n";
            $html .= "                if (btnText) {\n";
            $html .= "                    btnText.textContent = 'Enviar Respuesta';\n";
            $html .= "                }\n";
            $html .= "                btnEnviar.disabled = false;\n";
            $html .= "            }\n";
            $html .= "            \n";
            $html .= "            // Ocultar elementos relacionados\n";
            $html .= "            document.getElementById('fecha-respuesta-' + numeroFormateado).style.display = 'none';\n";
            $html .= "\n";
            $html .= "            document.getElementById('btn-eliminar-respuesta-' + numeroFormateado).style.display = 'none';\n";
            $html .= "            \n";
            $html .= "            // Ocultar mensaje de éxito\n";
            $html .= "            const mensajeExito = document.getElementById('mensaje-exito-respuesta-' + numeroFormateado);\n";
            $html .= "            if (mensajeExito) {\n";
            $html .= "                mensajeExito.style.display = 'none';\n";
            $html .= "            }\n";
            $html .= "            \n";
            $html .= "            alert('Respuesta eliminada exitosamente');\n";
            $html .= "            \n";
            $html .= "            // Recargar página para actualizar el estado del botón de término de actividad\n";
            $html .= "            setTimeout(() => {\n";
            $html .= "                window.location.reload();\n";
            $html .= "            }, 100);\n";
            $html .= "        } else {\n";
            $html .= "            alert('Error al eliminar: ' + result.message);\n";
            $html .= "        }\n";
            $html .= "    } catch (error) {\n";
            $html .= "        console.error('Error:', error);\n";
            $html .= "        alert('Error de conexión');\n";
            $html .= "    }\n";
            $html .= "}\n";
            $html .= "\n";
            $html .= "async function eliminarRetroalimentacionRespuestaLarga(numeroRespuesta, curso, unidad, actividad) {\n";
            $html .= "    if (!confirm('¿Estás seguro de que quieres eliminar la retroalimentación? Esta acción no se puede deshacer.')) {\n";
            $html .= "        return;\n";
            $html .= "    }\n";
            $html .= "    \n";
            $html .= "    const numeroFormateado = numeroRespuesta.toString().padStart(2, '0');\n";
            $html .= "    const textarea = document.getElementById('retro-respuesta-larga-' + numeroFormateado);\n";
            $html .= "    \n";
            $html .= "    try {\n";
            $html .= "        const response = await fetch('api/guardar_respuesta_larga.php', {\n";
            $html .= "            method: 'POST',\n";
            $html .= "            headers: { 'Content-Type': 'application/json' },\n";
            $html .= "            body: JSON.stringify({\n";
            $html .= "                tipo: 'eliminar_retroalimentacion',\n";
            $html .= "                numero_respuesta: numeroRespuesta,\n";
            $html .= "                curso: curso,\n";
            $html .= "                unidad: unidad,\n";
            $html .= "                actividad: actividad,\n";
            $html .= "                contenido: null\n";
            $html .= "            })\n";
            $html .= "        });\n";
            $html .= "        \n";
            $html .= "        const result = await response.json();\n";
            $html .= "        if (result.success) {\n";
            $html .= "            // Limpiar el campo y remover estilos\n";
            $html .= "            textarea.value = '';\n";
            $html .= "            textarea.classList.remove('guardado');\n";
            $html .= "            \n";
            $html .= "            // Ocultar elementos relacionados\n";
            $html .= "            document.getElementById('fecha-retroalimentacion-' + numeroFormateado).style.display = 'none';\n";
            $html .= "            document.getElementById('btn-eliminar-retroalimentacion-' + numeroFormateado).style.display = 'none';\n";
            $html .= "            \n";
            $html .= "            alert('Retroalimentación eliminada exitosamente');\n";
            $html .= "        } else {\n";
            $html .= "            alert('Error al eliminar: ' + result.message);\n";
            $html .= "        }\n";
            $html .= "    } catch (error) {\n";
            $html .= "        console.error('Error:', error);\n";
            $html .= "        alert('Error de conexión');\n";
            $html .= "    }\n";
            $html .= "}\n";
            $html .= "\n";
            $html .= "// Función para cargar datos existentes desde la base de datos\n";
            $html .= "async function cargarDatosExistentes() {\n";
            $html .= "    console.log('🔄 Cargando datos existentes...');\n";
            $html .= "    \n";
            $html .= "    try {\n";
            $html .= "        const response = await fetch('api/cargar_respuesta_larga.php', {\n";
            $html .= "            method: 'POST',\n";
            $html .= "            headers: { 'Content-Type': 'application/json' },\n";
            $html .= "            body: JSON.stringify({\n";
            $html .= "                numero_respuesta: {$numero_respuesta},\n";
            $html .= "                curso: '{$curso}',\n";
            $html .= "                unidad: '{$unidad}',\n";
            $html .= "                actividad: '{$actividad}'\n";
            $html .= "            })\n";
            $html .= "        });\n";
            $html .= "        \n";
            $html .= "        console.log('📡 Respuesta de carga:', response);\n";
            $html .= "        \n";
            $html .= "        if (!response.ok) {\n";
            $html .= "            throw new Error(`HTTP error! status: \${response.status}`);\n";
            $html .= "        }\n";
            $html .= "        \n";
            $html .= "        const result = await response.json();\n";
            $html .= "        console.log('📋 Resultado de carga:', result);\n";
            $html .= "        \n";
            $html .= "        if (result.success && result.data) {\n";
            $html .= "            // Cargar respuesta del estudiante\n";
            $html .= "            const textarea = document.getElementById('respuesta-estudiante-{$numero_formateado}');\n";
            $html .= "            if (textarea && result.data.respuesta) {\n";
            $html .= "                textarea.value = result.data.respuesta;\n";
            $html .= "                // Aplicar estilo de guardado al cargar datos existentes\n";
            $html .= "                textarea.classList.add('guardado');\n";
            $html .= "                console.log('✅ Respuesta cargada:', result.data.respuesta);\n";
            $html .= "                \n";
            $html .= "                // Deshabilitar botón y cambiar texto a 'Respuesta Enviada'\n";
                            $html .= "                const btnEnviar = document.getElementById('btn-enviar-respuesta-{$numero_formateado}');\n";
            $html .= "                if (btnEnviar) {\n";
            $html .= "                    const btnText = btnEnviar.querySelector('.btn-text');\n";
            $html .= "                    if (btnText) {\n";
            $html .= "                        btnText.textContent = 'Respuesta Enviada';\n";
            $html .= "                    }\n";
            $html .= "                    btnEnviar.disabled = true;\n";
            $html .= "                }\n";
            $html .= "                \n";
            $html .= "                // Mostrar mensaje de éxito\n";
            $html .= "                const mensajeExito = document.getElementById('mensaje-exito-respuesta-{$numero_formateado}');\n";
            $html .= "                if (mensajeExito) {\n";
            $html .= "                    mensajeExito.style.display = 'flex';\n";
            $html .= "                }\n";
            $html .= "                \n";
            $html .= "                // Mostrar fecha de última actualización\n";
            $html .= "                console.log('🔍 Fecha de respuesta recibida:', result.data.fecha_respuesta);\n";
            $html .= "                if (result.data.fecha_respuesta) {\n";
            $html .= "                    const fechaElement = document.getElementById('fecha-respuesta-{$numero_formateado}');\n";
            $html .= "                    console.log('🔍 Elemento de fecha encontrado:', fechaElement);\n";
            $html .= "                    if (fechaElement) {\n";
            $html .= "                        const fechaFormateada = 'Última actualización: ' + new Date(result.data.fecha_respuesta).toLocaleString();\n";
            $html .= "                        console.log('🔍 Mostrando fecha:', fechaFormateada);\n";
            $html .= "                        fechaElement.textContent = fechaFormateada;\n";
            $html .= "                        fechaElement.style.display = 'block';\n";
            $html .= "                    }\n";
            $html .= "                } else {\n";
            $html .= "                    console.log('⚠️ No hay fecha de respuesta en los datos');\n";
            $html .= "                }\n";
            $html .= "                \n";
            $html .= "                \n";
            $html .= "                // Mostrar botón de eliminar\n";
            $html .= "                const btnEliminar = document.getElementById('btn-eliminar-respuesta-{$numero_formateado}');\n";
            $html .= "                if (btnEliminar) {\n";
            $html .= "                    btnEliminar.style.display = 'inline-block';\n";
            $html .= "                }\n";
            $html .= "                \n";
            $html .= "                // Actualizar contador después de cargar\n";
            $html .= "                if (typeof actualizarContador === 'function') {\n";
            $html .= "                    actualizarContador();\n";
            $html .= "                }\n";
            $html .= "                \n";
            $html .= "                // También actualizar el contador con las palabras guardadas si están disponibles\n";
            $html .= "                if (result.data.palabras) {\n";
            $html .= "                    const contador = document.getElementById('contador-palabras-{$numero_formateado}');\n";
            $html .= "                    if (contador) {\n";
            $html .= "                        const palabras = parseInt(result.data.palabras);\n";
            $html .= "                        contador.textContent = palabras + ' palabras';\n";
            $html .= "                        console.log('📊 Contador actualizado con palabras guardadas:', palabras);\n";
            $html .= "                        \n";
            $html .= "                        // Aplicar color según el rango de palabras\n";
            $html .= "                        const minPalabras = {$min_palabras}; // Valor desde la configuración\n";
            $html .= "                        const maxPalabras = {$max_palabras}; // Valor desde la configuración\n";
            $html .= "                        \n";
            $html .= "                        if (palabras < minPalabras) {\n";
            $html .= "                            contador.style.color = '#ef4444'; // Rojo - insuficiente\n";
            $html .= "                            contador.style.fontWeight = 'bold';\n";
            $html .= "                        } else if (palabras > maxPalabras) {\n";
            $html .= "                            contador.style.color = '#ef4444'; // Rojo - excesivo\n";
            $html .= "                            contador.style.fontWeight = 'bold';\n";
            $html .= "                        } else {\n";
            $html .= "                            contador.style.color = '#10b981'; // Verde - rango ideal\n";
            $html .= "                            contador.style.fontWeight = 'bold';\n";
            $html .= "                        }\n";
            $html .= "                    }\n";
            $html .= "                }\n";
            $html .= "            }\n";
            $html .= "            \n";
            $html .= "            // Cargar retroalimentación\n";
            $html .= "            const textareaRetro = document.getElementById('retro-respuesta-larga-{$numero_formateado}');\n";
            $html .= "            if (textareaRetro && result.data.retroalimentacion) {\n";
            $html .= "                textareaRetro.value = result.data.retroalimentacion;\n";
            $html .= "                // Aplicar estilo de guardado al cargar datos existentes\n";
            $html .= "                textareaRetro.classList.add('guardado');\n";
            $html .= "                console.log('✅ Retroalimentación cargada:', result.data.retroalimentacion);\n";
            $html .= "                \n";
            $html .= "                // Mostrar fecha de retroalimentación\n";
            $html .= "                console.log('🔍 Fecha de retroalimentación recibida:', result.data.fecha_retroalimentacion);\n";
            $html .= "                if (result.data.fecha_retroalimentacion) {\n";
            $html .= "                    const fechaRetro = document.getElementById('fecha-retroalimentacion-{$numero_formateado}');\n";
            $html .= "                    if (fechaRetro) {\n";
            $html .= "                        const fechaFormateada = 'Última actualización: ' + new Date(result.data.fecha_retroalimentacion).toLocaleString();\n";
            $html .= "                        fechaRetro.textContent = fechaFormateada;\n";
            $html .= "                        fechaRetro.style.display = 'block';\n";
            $html .= "                        console.log('✅ Fecha de retroalimentación mostrada correctamente');\n";
            $html .= "                    } else {\n";
            $html .= "                        console.error('❌ No se encontró el elemento fecha-retroalimentacion-{$numero_formateado}');\n";
            $html .= "                    }\n";
            $html .= "                }\n";
            $html .= "                \n";
            $html .= "                // Mostrar botón de eliminar retroalimentación\n";
            $html .= "                const btnEliminarRetro = document.getElementById('btn-eliminar-retroalimentacion-{$numero_formateado}');\n";
            $html .= "                if (btnEliminarRetro) {\n";
            $html .= "                    btnEliminarRetro.style.display = 'inline-block';\n";
            $html .= "                }\n";
            $html .= "            }\n";
            $html .= "            \n";
            $html .= "            // Cargar nota\n";
            $html .= "            const inputNota = document.getElementById('calificacion-respuesta-larga-{$numero_formateado}');\n";
            $html .= "            if (inputNota && result.data.nota) {\n";
            $html .= "                inputNota.value = result.data.nota;\n";
            $html .= "                // Aplicar estilo de guardado al cargar datos existentes\n";
            $html .= "                inputNota.classList.add('guardado');\n";
            $html .= "                console.log('✅ Nota cargada:', result.data.nota);\n";
            $html .= "                \n";
            $html .= "                // Mostrar fecha de nota\n";
            $html .= "                console.log('🔍 Fecha de nota recibida:', result.data.fecha_nota);\n";
            $html .= "                if (result.data.fecha_nota) {\n";
            $html .= "                    const fechaNota = document.getElementById('fecha-nota-{$numero_formateado}');\n";
            $html .= "                    if (fechaNota) {\n";
            $html .= "                        const fechaFormateada = 'Última actualización: ' + new Date(result.data.fecha_nota).toLocaleString();\n";
            $html .= "                        fechaNota.textContent = fechaFormateada;\n";
            $html .= "                        fechaNota.style.display = 'block';\n";
            $html .= "                        console.log('✅ Fecha de nota mostrada correctamente');\n";
            $html .= "                    } else {\n";
            $html .= "                        console.error('❌ No se encontró el elemento fecha-nota-{$numero_formateado}');\n";
            $html .= "                    }\n";
            $html .= "                }\n";
            $html .= "            }\n";
            $html .= "            \n";
            $html .= "            // Mostrar fecha límite si existe\n";
            $html .= "            if (result.data.fecha_limite) {\n";
            $html .= "                const fechaLimiteContainer = document.getElementById('fecha-limite-{$numero_formateado}');\n";
            $html .= "                const fechaLimiteValue = document.getElementById('fecha-limite-value-{$numero_formateado}');\n";
            $html .= "                if (fechaLimiteContainer && fechaLimiteValue) {\n";
            $html .= "                    const fechaLimite = new Date(result.data.fecha_limite);\n";
            $html .= "                    const ahora = new Date();\n";
            $html .= "                    fechaLimiteValue.textContent = fechaLimite.toLocaleString();\n";
            $html .= "                    \n";
            $html .= "                    // Verificar si la fecha límite ha pasado\n";
            $html .= "                    if (ahora > fechaLimite) {\n";
            $html .= "                        fechaLimiteContainer.classList.add('fecha-limite-vencida');\n";
            $html .= "                        fechaLimiteValue.textContent += ' (VENCIDA)';\n";
            $html .= "                    }\n";
            $html .= "                    \n";
            $html .= "                    fechaLimiteContainer.style.display = 'block';\n";
            $html .= "                }\n";
            $html .= "            }\n";
            $html .= "            \n";
            $html .= "            console.log('✅ Todos los datos cargados exitosamente');\n";
            $html .= "        } else {\n";
            $html .= "            console.log('ℹ️ No hay datos guardados o error:', result.message);\n";
            $html .= "        }\n";
            $html .= "    } catch (error) {\n";
            $html .= "        console.error('❌ Error al cargar datos:', error);\n";
            $html .= "        // No mostrar alerta al usuario, solo log en consola\n";
            $html .= "    }\n";
            $html .= "}\n";
            $html .= "\n";
            
            // Agregar lógica para guardar fecha límite inicial si está configurada
            if ($incluir_fecha_limite && !empty($fecha_limite)) {
                $html .= "// Guardar fecha límite inicial si está configurada\n";
                $html .= "async function guardarFechaLimiteInicial() {\n";
                $html .= "    console.log('🕒 Guardando fecha límite inicial...');\n";
                $html .= "    \n";
                $html .= "    try {\n";
                $html .= "        const response = await fetch('api/guardar_respuesta_larga.php', {\n";
                $html .= "            method: 'POST',\n";
                $html .= "            headers: { 'Content-Type': 'application/json' },\n";
                $html .= "            body: JSON.stringify({\n";
                $html .= "                tipo: 'fecha_limite',\n";
                $html .= "                numero_respuesta: {$numero_respuesta},\n";
                $html .= "                curso: '{$curso}',\n";
                $html .= "                unidad: '{$unidad}',\n";
                $html .= "                actividad: '{$actividad}',\n";
                $html .= "                contenido: '{$fecha_limite}'\n";
                $html .= "            })\n";
                $html .= "        });\n";
                $html .= "        \n";
                $html .= "        const result = await response.json();\n";
                $html .= "        if (result.success) {\n";
                $html .= "            console.log('✅ Fecha límite inicial guardada');\n";
                $html .= "        } else {\n";
                $html .= "            console.log('⚠️ Error al guardar fecha límite inicial:', result.message);\n";
                $html .= "        }\n";
                $html .= "    } catch (error) {\n";
                $html .= "        console.error('❌ Error al guardar fecha límite inicial:', error);\n";
                $html .= "    }\n";
                $html .= "}\n";
                $html .= "\n";
                $html .= "// Llamar a la función al cargar la página\n";
                $html .= "document.addEventListener('DOMContentLoaded', function() {\n";
                $html .= "    guardarFechaLimiteInicial();\n";
                $html .= "});\n";
            }
            
            $html .= "</script>\n";
            break;
            
        case 'seccion-cargar-archivo':
            $html .= "    <h3>Cargar Archivo</h3>\n";
            $html .= "    \n";
            $html .= "    <div class=\"cargar-archivo-container\">\n";
            $html .= "        <div class=\"cargar-archivo-titulo\">Subir Documento de Trabajo</div>\n";
            $html .= "        <div class=\"cargar-archivo-descripcion\">\n";
            $html .= "            Sube tu documento de trabajo en formato PDF o Word. Asegúrate de que el archivo contenga tu respuesta completa a la actividad.\n";
            $html .= "        </div>\n";
            $html .= "        \n";
            $html .= "        <div class=\"upload-area\">\n";
            $html .= "            <div class=\"upload-icon\">📁</div>\n";
            $html .= "            <div class=\"upload-text\">Arrastra y suelta tu archivo aquí</div>\n";
            $html .= "            <div class=\"upload-subtext\">o haz clic para seleccionar un archivo</div>\n";
            $html .= "            <button class=\"btn-seleccionar\">\n";
            $html .= "                <i class=\"fas fa-upload\"></i>\n";
            $html .= "                Seleccionar Archivo\n";
            $html .= "            </button>\n";
            $html .= "            <input type=\"file\" class=\"file-input\" accept=\".pdf,.doc,.docx\">\n";
            $html .= "        </div>\n";
            $html .= "        \n";
            $html .= "        <div class=\"formatos-soportados\">\n";
            $html .= "            <span class=\"formato-badge\">PDF</span>\n";
            $html .= "            <span class=\"formato-badge\">DOC</span>\n";
            $html .= "            <span class=\"formato-badge\">DOCX</span>\n";
            $html .= "        </div>\n";
            $html .= "        \n";
            $html .= "        <div class=\"archivos-subidos\" id=\"archivos-subidos\">\n";
            $html .= "            <!-- Los archivos subidos aparecerán aquí -->\n";
            $html .= "        </div>\n";
            $html .= "        \n";
            $html .= "        <button class=\"btn-subir-documentos\" id=\"btn-subir-documentos\" disabled>\n";
            $html .= "            <i class=\"fas fa-cloud-upload-alt\"></i>\n";
            $html .= "            Subir Documento(s)\n";
            $html .= "        </button>\n";
            $html .= "    </div>\n";
            break;
            
            
        case 'seccion-foro':
            $html .= "    <h3>Foro de Opinión</h3>\n";
            $html .= "    \n";
            $html .= "    <div class=\"foro-container\">\n";
            $html .= "        <div class=\"foro-titulo\">¿Qué opinas sobre esta actividad?</div>\n";
            $html .= "        <div class=\"foro-descripcion\">\n";
            $html .= "            Comparte tu experiencia y opinión sobre esta actividad. ¿Te ha resultado útil? ¿Qué aspectos destacarías?\n";
            $html .= "        </div>\n";
            $html .= "        \n";
            $html .= "        <div class=\"foro-tema\">\n";
            $html .= "            <div class=\"tema-header\">\n";
            $html .= "                <div class=\"tema-avatar\">P</div>\n";
            $html .= "                <div class=\"tema-info\">\n";
            $html .= "                    <div class=\"tema-autor\">Prof. María González</div>\n";
            $html .= "                    <div class=\"tema-fecha\">Hace 2 horas</div>\n";
            $html .= "                </div>\n";
            $html .= "            </div>\n";
            $html .= "            <div class=\"tema-contenido\">\n";
            $html .= "                Esta actividad está diseñada para ayudarte a comprender mejor los conceptos. ¿Cuál ha sido tu experiencia hasta ahora?\n";
            $html .= "            </div>\n";
            $html .= "            <div class=\"tema-acciones\">\n";
            $html .= "                <button class=\"btn-me-gusta\">\n";
            $html .= "                    <i class=\"fas fa-heart\"></i>\n";
            $html .= "                    <span>0</span>\n";
            $html .= "                </button>\n";
            $html .= "                <button class=\"btn-responder\">\n";
            $html .= "                    <i class=\"fas fa-reply\"></i>\n";
            $html .= "                    Responder\n";
            $html .= "                </button>\n";
            $html .= "            </div>\n";
            $html .= "        </div>\n";
            $html .= "        \n";
            $html .= "        <div class=\"foro-nueva-respuesta\" id=\"nueva-respuesta\" style=\"display: none;\">\n";
            $html .= "            <h4>Tu Respuesta</h4>\n";
            $html .= "            <textarea class=\"textarea-respuesta-foro\" id=\"texto-respuesta\" placeholder=\"Escribe tu opinión sobre la actividad...\"></textarea>\n";
            $html .= "            <button class=\"btn-publicar\">\n";
            $html .= "                <i class=\"fas fa-paper-plane\"></i>\n";
            $html .= "                Publicar Respuesta\n";
            $html .= "            </button>\n";
            $html .= "        </div>\n";
            $html .= "    </div>\n";
            break;
            
        case 'termino-actividad':
            $textoBoton = isset($seccion['contenido']['texto_boton']) ? $seccion['contenido']['texto_boton'] : 'Marcar como Completada';
            $variableCondicion = isset($seccion['contenido']['variable_condicion']) ? $seccion['contenido']['variable_condicion'] : '';
            $mensajeExito = isset($seccion['contenido']['mensaje_exito']) ? $seccion['contenido']['mensaje_exito'] : '¡Actividad completada exitosamente!';
            
            // Obtener número de actividad
            global $actividad;
            $numeroActividadTermino = $actividad ? intval($actividad) : intval($numero);
            
            // Normalizar unidad para el nombre de la tabla
            $unidadParaTabla = preg_replace('/Unidad\s*0*(\d+)/i', '$1', $unidad);
            if (strlen($unidadParaTabla) == 1) {
                $unidadParaTabla = '0' . $unidadParaTabla;
            }
            
            $html .= "    <div class=\"termino-actividad-container\">\n";
            $html .= "        <button class=\"btn-terminar-actividad\" id=\"btn-terminar-act-" . $numeroActividadTermino . "\" onclick=\"marcarActividadComoCompletada(" . $numeroActividadTermino . ", '" . htmlspecialchars($variableCondicion) . "', '" . htmlspecialchars($curso) . "', 'Unidad " . htmlspecialchars($unidadParaTabla) . "')\" disabled>\n";
            $html .= "            <i class=\"fas fa-check-circle\"></i>\n";
            $html .= "            " . htmlspecialchars($textoBoton) . "\n";
            $html .= "        </button>\n";
            $html .= "        <div id=\"mensaje-completada-" . $numeroActividadTermino . "\" style=\"display: none; margin-top: 1rem; padding: 1rem; background: #dbeafe; border: 1px solid #3b82f6; border-radius: 8px; color: #1e40af; text-align: center;\">\n";
            $html .= "            <i class=\"fas fa-arrow-right\" style=\"font-size: 1.5rem; margin-bottom: 0.5rem; display: block;\"></i>\n";
            $html .= "            <span style=\"font-weight: 600;\">¡Actividad completada!</span>\n";
            $html .= "            <div style=\"margin-top: 0.5rem; font-size: 0.9rem;\">Puedes continuar con la siguiente actividad</div>\n";
            $html .= "        </div>\n";
            $html .= "    </div>\n";
            
            // Agregar JavaScript para verificar condición y actualizar botón
            $html .= "<script>\n";
            $html .= "document.addEventListener('DOMContentLoaded', function() {\n";
            $html .= "    mostrarBotonTerminoActividad(" . $numeroActividadTermino . ", '" . htmlspecialchars($variableCondicion) . "', '" . htmlspecialchars($curso) . "', '" . htmlspecialchars($unidadParaTabla) . "');\n";
            $html .= "});\n";
            $html .= "\n";
            $html .= "function mostrarBotonTerminoActividad(numeroActividad, variableCondicion, curso, unidad) {\n";
            $html .= "    const btn = document.getElementById('btn-terminar-act-' + numeroActividad);\n";
            $html .= "    const nombreTabla = 'avance_' + curso + '_u' + unidad;\n";
            $html .= "    const numeroActividadFormateado = numeroActividad.toString().padStart(2, '0');\n";
            $html .= "    const variableACT = 'ACT' + numeroActividadFormateado;\n";
            $html .= "    \n";
            $html .= "    console.log('🔍 DEBUG - mostrarBotonTerminoActividad');\n";
            $html .= "    console.log('  - numeroActividad:', numeroActividad);\n";
            $html .= "    console.log('  - variableCondicion:', variableCondicion);\n";
            $html .= "    console.log('  - curso:', curso);\n";
            $html .= "    console.log('  - unidad:', unidad);\n";
            $html .= "    console.log('  - nombreTabla:', nombreTabla);\n";
            $html .= "    console.log('  - variableACT:', variableACT);\n";
            $html .= "    \n";
            $html .= "    // Verificar si la actividad ya está completada (ACTXX = 1)\n";
            $html .= "    fetch('api/verificar_condicion_interna.php', {\n";
            $html .= "        method: 'POST',\n";
            $html .= "        headers: { 'Content-Type': 'application/json' },\n";
            $html .= "        body: JSON.stringify({\n";
            $html .= "            variable: variableACT,\n";
            $html .= "            tabla: nombreTabla\n";
            $html .= "        })\n";
            $html .= "    })\n";
            $html .= "    .then(response => response.json())\n";
            $html .= "    .then(dataACT => {\n";
            $html .= "        const valorACT = parseInt(dataACT.valor) || 0;\n";
            $html .= "        if (dataACT.success && valorACT === 1) {\n";
            $html .= "            // Si ACTXX = 1, desactivar el botón y mostrar mensaje de completada\n";
            $html .= "            btn.disabled = true;\n";
            $html .= "            const mensajeCompletada = document.getElementById('mensaje-completada-' + numeroActividad);\n";
            $html .= "            if (mensajeCompletada) {\n";
            $html .= "                mensajeCompletada.style.display = 'block';\n";
            $html .= "            }\n";
            $html .= "            return;\n";
            $html .= "        }\n";
            $html .= "        \n";
            $html .= "        // Si la actividad no está completada, verificar la condición\n";
            $html .= "        return fetch('api/verificar_condicion_interna.php', {\n";
            $html .= "            method: 'POST',\n";
            $html .= "            headers: { 'Content-Type': 'application/json' },\n";
            $html .= "            body: JSON.stringify({\n";
            $html .= "                variable: variableCondicion,\n";
            $html .= "                tabla: nombreTabla\n";
            $html .= "            })\n";
            $html .= "        });\n";
            $html .= "    })\n";
            $html .= "    .then(response => {\n";
            $html .= "        if (!response) return;\n";
            $html .= "        return response.json();\n";
            $html .= "    })\n";
            $html .= "    .then(data => {\n";
            $html .= "        if (!data) return;\n";
            $html .= "        console.log('📊 DEBUG - Respuesta del servidor:', data);\n";
            $html .= "        const valor = parseInt(data.valor) || 0;\n";
            $html .= "        if (data.success && valor === 1) {\n";
            $html .= "            btn.disabled = false;\n";
            $html .= "        } else {\n";
            $html .= "            btn.disabled = true;\n";
            $html .= "        }\n";
            $html .= "    })\n";
            $html .= "    .catch(error => {\n";
            $html .= "        console.error('❌ Error:', error);\n";
            $html .= "        btn.disabled = true;\n";
            $html .= "    });\n";
            $html .= "}\n";
            $html .= "\n";
            $html .= "async function marcarActividadComoCompletada(numeroActividad, variableCondicion, curso, unidad) {\n";
            $html .= "    const btn = document.getElementById('btn-terminar-act-' + numeroActividad);\n";
            $html .= "    const mensajeCompletada = document.getElementById('mensaje-completada-' + numeroActividad);\n";
            $html .= "    \n";
            $html .= "    try {\n";
            $html .= "        const response = await fetch('api/marcar_actividad_completada.php?curso=' + curso + '&unidad=' + encodeURIComponent(unidad), {\n";
            $html .= "            method: 'POST',\n";
            $html .= "            headers: { 'Content-Type': 'application/json' },\n";
            $html .= "            body: JSON.stringify({ numeroActividad: numeroActividad, variable: variableCondicion })\n";
            $html .= "        });\n";
            $html .= "        \n";
            $html .= "        const data = await response.json();\n";
            $html .= "        \n";
            $html .= "        if (data.success) {\n";
            $html .= "            if (mensajeCompletada) mensajeCompletada.style.display = 'block';\n";
            $html .= "            if (btn) btn.disabled = true;\n";
            $html .= "            // Recargar la página para actualizar el botón 'Siguiente'\n";
            $html .= "            window.location.reload();\n";
            $html .= "        } else {\n";
            $html .= "            alert('Error: ' + (data.message || 'Error desconocido'));\n";
            $html .= "        }\n";
            $html .= "    } catch (error) {\n";
            $html .= "        console.error('Error:', error);\n";
            $html .= "        alert('Error de conexión');\n";
            $html .= "    }\n";
            $html .= "}\n";
            $html .= "</script>\n";
            break;
            
        case 'texto-rico':
            $titulo = isset($seccion['contenido']['titulo']) ? htmlspecialchars($seccion['contenido']['titulo']) : '';
            $contenido = isset($seccion['contenido']['contenido']) ? $seccion['contenido']['contenido'] : '';
            
            $html .= "    <div class=\"texto-rico-content\">\n";
            if (!empty($titulo)) {
                $html .= "        <h3 class=\"texto-rico-titulo\">" . $titulo . "</h3>\n";
            }
            $html .= "        <div class=\"texto-rico-contenido\">\n";
            $html .= "            " . $contenido . "\n";
            $html .= "        </div>\n";
            $html .= "    </div>\n";
            break;
            
        case 'ruta-ejercitacion':
            $titulo = isset($seccion['contenido']['titulo']) ? htmlspecialchars($seccion['contenido']['titulo']) : '';
            $archivo = isset($seccion['contenido']['archivo']) ? $seccion['contenido']['archivo'] : '';
            
            $html .= "    <div class=\"ejercitacion-container\">\n";
            if (!empty($titulo)) {
                $html .= "        <h3 class=\"ejercitacion-titulo\">" . $titulo . "</h3>\n";
            }
            if (!empty($archivo)) {
                // Si hay un archivo, cargarlo en un iframe
                $html .= "        <div class=\"ejercitacion-iframe-container\">\n";
                $html .= "            <iframe src=\"rutas/" . htmlspecialchars($archivo) . "\" class=\"ejercitacion-iframe\" style=\"width: 100%; height: 800px; border: none; border-radius: 8px; box-shadow: 0 2px 8px rgba(0,0,0,0.1);\"></iframe>\n";
                $html .= "        </div>\n";
            } else {
                // Si no hay archivo, mostrar mensaje
                $html .= "        <div class=\"ejercitacion-placeholder\">\n";
                $html .= "            <p style=\"text-align: center; color: #666; padding: 2rem;\">No se ha cargado un archivo de ejercicios aún.</p>\n";
                $html .= "        </div>\n";
            }
            $html .= "    </div>\n";
            break;
            
        case 'geogebra-applet':
            $titulo = isset($seccion['contenido']['titulo']) ? htmlspecialchars($seccion['contenido']['titulo']) : '';
            $iframeCode = isset($seccion['contenido']['url']) ? $seccion['contenido']['url'] : '';
            
            $html .= "    <div class=\"geogebra-container\">\n";
            if (!empty($titulo)) {
                $html .= "        <h3 class=\"geogebra-titulo\">" . $titulo . "</h3>\n";
            }
            if (!empty($iframeCode)) {
                // Si hay código de iframe, renderizarlo directamente
                $html .= "        <div class=\"geogebra-iframe-container\">\n";
                $html .= "            " . $iframeCode . "\n";
                $html .= "        </div>\n";
            } else {
                // Si no hay código, mostrar mensaje
                $html .= "        <div class=\"geogebra-placeholder\">\n";
                $html .= "            <p style=\"text-align: center; color: #666; padding: 2rem;\">No se ha configurado un applet de GeoGebra aún.</p>\n";
                $html .= "        </div>\n";
            }
            $html .= "    </div>\n";
            break;
            
        default:
            $html .= "    <p>Contenido de la sección " . $seccion['nombre'] . "</p>\n";
            break;
    }
    
    $html .= "</section>\n\n";
    
    return $html;
}

// Función para normalizar el formato de unidad
function normalizarUnidad($unidad) {
    // Convertir "Unidad 01" a "Unidad 1", "Unidad 02" a "Unidad 2", etc.
    return preg_replace('/Unidad 0*(\d+)/', 'Unidad $1', $unidad);
}


// Función para obtener el aprendizaje esperado desde la configuración
function obtenerAprendizajeEsperadoConfiguracion($curso = null, $unidad = null) {
    global $cursos;
    
    // Usar parámetros pasados o los de la URL
    $curso = $curso ?: (isset($_POST['curso']) ? $_POST['curso'] : '');
    $unidad = $unidad ?: (isset($_POST['unidad']) ? $_POST['unidad'] : '');
    
    // Normalizar formato de unidad
    $unidad_normalizada = normalizarUnidad($unidad);
    
    if ($curso && $unidad_normalizada && isset($cursos[$curso]['unidades'][$unidad_normalizada])) {
        $aprendizaje_esperado = $cursos[$curso]['unidades'][$unidad_normalizada]['aprendizaje_esperado'];
        return [
            'titulo' => 'Aprendizaje Esperado',
            'contenido' => $aprendizaje_esperado,
            'fuente' => 'configuracion_cursos.php'
        ];
    }
    
    // Fallback si no se encuentra en configuración
    return [
        'titulo' => 'Aprendizaje Esperado',
        'contenido' => 'Reconocer y manejar la interfaz de la actividad, identificando sus principales elementos y funcionalidades.',
        'fuente' => 'default'
    ];
}

// Función para obtener los criterios de evaluación desde la configuración
function obtenerCriteriosEvaluacionConfiguracion($curso = null, $unidad = null) {
    global $cursos;
    
    // Usar parámetros pasados o los de la URL
    $curso = $curso ?: (isset($_POST['curso']) ? $_POST['curso'] : '');
    $unidad = $unidad ?: (isset($_POST['unidad']) ? $_POST['unidad'] : '');
    
    // Normalizar formato de unidad
    $unidad_normalizada = normalizarUnidad($unidad);
    
    if ($curso && $unidad_normalizada && isset($cursos[$curso]['unidades'][$unidad_normalizada])) {
        $criterios_config = $cursos[$curso]['unidades'][$unidad_normalizada]['criterios_evaluacion'];
        
        // Convertir array asociativo a array de objetos con código y descripción
        $criterios = [];
        foreach ($criterios_config as $codigo => $descripcion) {
            $criterios[] = [
                'codigo' => $codigo,
                'descripcion' => $descripcion
            ];
        }
        
        return [
            'titulo' => 'Criterios de Evaluación',
            'criterios' => $criterios,
            'fuente' => 'configuracion_cursos.php'
        ];
    }
    
    // Fallback si no se encuentra en configuración
    return [
        'titulo' => 'Criterios de Evaluación',
        'criterios' => [
            [
                'codigo' => '1.1.1',
                'descripcion' => 'Identificar correctamente los elementos principales de la actividad.'
            ],
            [
                'codigo' => '1.1.2',
                'descripcion' => 'Completar la actividad siguiendo las instrucciones proporcionadas.'
            ]
        ],
        'fuente' => 'default'
    ];
}

// Función para obtener los contenidos desde la configuración
function obtenerContenidosConfiguracion($curso = null, $unidad = null) {
    global $cursos;
    
    // Usar parámetros pasados o los de la URL
    $curso = $curso ?: (isset($_POST['curso']) ? $_POST['curso'] : '');
    $unidad = $unidad ?: (isset($_POST['unidad']) ? $_POST['unidad'] : '');
    
    // Normalizar formato de unidad
    $unidad_normalizada = normalizarUnidad($unidad);
    
    if ($curso && $unidad_normalizada && isset($cursos[$curso]['unidades'][$unidad_normalizada])) {
        $contenidos_config = $cursos[$curso]['unidades'][$unidad_normalizada]['contenidos'];
        
        // Los contenidos ya están en el formato correcto
        return [
            'titulo' => 'Contenidos',
            'contenidos' => $contenidos_config,
            'fuente' => 'configuracion_cursos.php'
        ];
    }
    
    // Fallback si no se encuentra en configuración
    return [
        'titulo' => 'Contenidos',
        'contenidos' => [
            [
                'titulo' => 'Conceptos básicos',
                'descripcion' => 'Introducción a los conceptos fundamentales de la actividad.',
                'icono' => 'fas fa-book'
            ],
            [
                'titulo' => 'Aplicación práctica',
                'descripcion' => 'Ejercicios y actividades para aplicar los conceptos aprendidos.',
                'icono' => 'fas fa-tasks'
            ]
        ],
        'fuente' => 'default'
    ];
}
?>
