<?php
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST');
header('Access-Control-Allow-Headers: Content-Type');

// Incluir archivos de configuración
require_once '../config.php';
require_once '../conexion.php';

// Función para generar el archivo principal de la ruta
function generarArchivoRuta($ruta_nombre, $niveles) {
    // Extraer información del nombre de la ruta (ej: ruta_c001_u01_act04)
    preg_match('/ruta_c(\d+)_u(\d+)_act(\d+)/', $ruta_nombre, $matches);
    $curso = 'c' . $matches[1];
    $unidad = str_pad($matches[2], 2, '0', STR_PAD_LEFT);
    $actividad = str_pad($matches[3], 2, '0', STR_PAD_LEFT);
    
    // Generar configuración de niveles para JavaScript
    $levels_js = '';
    foreach ($niveles as $index => $nivel) {
        $numero = $index + 1;
        $carpeta = $nivel['carpeta'];
        $puntos = $nivel['puntos_requeridos'];
        
        // No incluimos 'items' porque se contará dinámicamente en el cliente
        $levels_js .= "            {$numero}: { requiredPoints: {$puntos}, folder: '{$carpeta}' },\n";
    }
    $levels_js = rtrim($levels_js, ",\n");
    
    $contenido = "<!DOCTYPE html>
<html lang=\"es\">
<head>
    <meta charset=\"UTF-8\">
    <meta name=\"viewport\" content=\"width=device-width, initial-scale=1.0\">
    <title>Ruta de Ejercitación Progresiva</title>
    <link rel=\"stylesheet\" href=\"https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css\">
    <link rel=\"stylesheet\" href=\"item.css\">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: white;
            min-height: 100vh;
            padding: 2rem;
        }

        .rutas-container {
            max-width: 1200px;
            margin: 0 auto;
        }

        .stats-header {
            background: linear-gradient(135deg, #1e40af 0%, #1e3a8a 100%);
            border-radius: 20px 20px 0 0;
            padding: 1.5rem;
            display: flex;
            flex-direction: column;
            justify-content: center;
            align-items: center;
            gap: 1rem;
            position: relative;
            overflow: hidden;
        }

        .stats-header::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: linear-gradient(135deg, rgba(255,255,255,0.1) 0%, rgba(255,255,255,0.05) 100%);
            pointer-events: none;
        }

        .stat-item-center,
        .stat-item-right {
            position: relative;
            z-index: 1;
        }

        .stat-item-left {
            text-align: left;
        }

        .stat-item-center {
            text-align: center;
            padding: 0 1rem;
        }

        .stat-item-right {
            text-align: right;
        }

        .stat-item {
            flex: 1;
            min-width: 200px;
            text-align: center;
        }

        .stat-label {
            color: rgba(255, 255, 255, 0.9);
            font-size: 0.75rem;
            margin-bottom: 0.5rem;
            text-transform: uppercase;
            letter-spacing: 1px;
            font-weight: 600;
        }

        .stat-value {
            color: white;
            font-size: 2rem;
            font-weight: 900;
            text-shadow: 0 2px 10px rgba(0, 0, 0, 0.2);
            line-height: 1;
        }

        .stars-container {
            display: flex;
            gap: 0.5rem;
            justify-content: center;
            position: relative;
            z-index: 1;
        }

        .star {
            font-size: 3rem;
            color: rgba(255, 255, 255, 0.3);
            transition: all 0.5s ease;
            filter: drop-shadow(0 2px 8px rgba(0, 0, 0, 0.3));
        }

        .star.earned {
            color: #fbbf24;
            text-shadow: 0 0 30px rgba(251, 191, 36, 0.8);
            transform: scale(1.1);
            filter: drop-shadow(0 8px 20px rgba(251, 191, 36, 0.6));
        }

        .current-level-title {
            color: white;
            font-size: 1.2rem;
            font-weight: 700;
            text-align: center;
            position: relative;
            z-index: 1;
            margin-bottom: 0.5rem;
            text-transform: uppercase;
            letter-spacing: 1px;
        }

        .progress-bar-container {
            background: rgba(255, 255, 255, 0.2);
            height: 12px;
            border-radius: 12px;
            overflow: hidden;
            position: relative;
            backdrop-filter: blur(10px);
            width: 100%;
            max-width: 400px;
        }

        .progress-bar {
            height: 100%;
            background: linear-gradient(90deg, #ffd700 0%, #ffed4e 100%);
            transition: width 0.5s ease;
            display: flex;
            align-items: center;
            justify-content: flex-end;
            padding-right: 4px;
            color: #764ba2;
            font-weight: 800;
            font-size: 0.6rem;
            box-shadow: 0 2px 8px rgba(255, 215, 0, 0.5);
        }

        .progress-text {
            color: #764ba2;
            font-weight: 800;
            font-size: 0.6rem;
            white-space: nowrap;
        }

        .level-message {
            color: white;
            font-size: 0.9rem;
            font-weight: 600;
            text-align: center;
            position: relative;
            z-index: 1;
            margin-top: 0.5rem;
        }

        .exercise-container {
            background: white;
            border-radius: 20px;
            padding: 2rem;
            box-shadow: 0 10px 40px rgba(0, 0, 0, 0.1);
            margin-top: 2rem;
            min-height: 500px;
        }

        .exercise-iframe {
            width: 100%;
            height: 600px;
            border: none;
            border-radius: 12px;
            box-shadow: 0 4px 20px rgba(0, 0, 0, 0.1);
        }

        .debug-section {
            background: rgba(0,0,0,0.3);
            padding: 1rem;
            border-radius: 10px;
            margin-bottom: 1rem;
            font-family: 'Courier New', monospace;
            font-size: 0.9rem;
        }

        @keyframes slideDown {
            from {
                opacity: 0;
                transform: translateY(-10px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }
    </style>
</head>
<body>
    <div class=\"rutas-container\">
        <div class=\"stats-header\">
            <div class=\"current-level-title\" id=\"current-level-title\">Nivel Inicial</div>
            <div class=\"stars-container\">
                <span class=\"star\" id=\"star-1\">⭐</span>
                <span class=\"star\" id=\"star-2\">⭐</span>
                <span class=\"star\" id=\"star-3\">⭐</span>
            </div>
            <div class=\"progress-bar-container\">
                <div class=\"progress-bar\" id=\"progress-bar\" style=\"width: 0%;\">
                    <span class=\"progress-text\" id=\"progress-text\">0%</span>
                </div>
            </div>
            <div class=\"level-message\" id=\"level-message\">Vamos rumbo al Nivel Intermedio</div>
        </div>

        <!-- Mensaje temporal de información del ejercicio -->
        <div id=\"exercise-info\" style=\"display: none; background: linear-gradient(135deg, #10b981 0%, #059669 100%); color: white; padding: 1rem; border-radius: 0 0 20px 20px; text-align: center; font-size: 0.9rem; font-weight: 600;\">
            <div style=\"display: flex; justify-content: center; gap: 2rem; flex-wrap: wrap; margin-bottom: 0.75rem;\">
                <span id=\"info-points\">Puntaje Acumulado: <strong>0</strong></span>
                <span id=\"info-item\">Item: <strong>-</strong></span>
                <span id=\"info-level\">Nivel: <strong>1</strong></span>
                <span id=\"info-attempt\">Intento: <strong>0</strong></span>
            </div>
            <div style=\"display: flex; justify-content: center; gap: 1rem; flex-wrap: wrap; font-size: 0.85rem;\">
                <span id=\"progress-nivel1\">N1: 0/0</span>
                <span id=\"progress-nivel2\">N2: 0/0</span>
                <span id=\"progress-nivel3\">N3: 0/0</span>
            </div>
        </div>

        <div class=\"exercise-container\">
            <iframe id=\"exercise-iframe\" class=\"exercise-iframe\" src=\"\"></iframe>
        </div>
    </div>

    <script>
        // Configuración de niveles
        const levels = {
{$levels_js}
        };

        let gameState = {
            totalPoints: 0,
            currentLevel: 1,
            stars: 0,
            usedItems: [],
            totalAttempts: 0 // Contador global de intentos
        };
        
        let currentItem = null; // Item actual que se está mostrando
        let currentItemPoints = 0; // Puntos del item actual
        let attemptNumber = 0; // Número de intento en el item actual

        // Cargar estado desde la base de datos
        async function loadState() {
            try {
                console.log('Cargando estado desde BD...');
                const response = await fetch(`../../api/obtener_estado_ruta.php?curso={$curso}&unidad={$unidad}&actividad={$actividad}`);
                const data = await response.json();
                console.log('Respuesta de la API:', data);
                
                if (data.success) {
                    gameState.stars = parseInt(data.data.estrellas, 10) || 0;
                    gameState.totalPoints = parseInt(data.data.puntos, 10) || 0;
                    gameState.currentLevel = parseInt(data.data.nivelActual, 10) || 1;
                    gameState.totalAttempts = parseInt(data.data.intentos, 10) || 0;
                    console.log('✅ GameState sincronizado con BD:', gameState);
                } else {
                    console.log('⚠️ No se pudo cargar desde BD, usando valores por defecto');
                }
            } catch (error) {
                console.error('❌ Error al cargar estado desde BD:', error);
            }
        }

        // Guardar estado en la base de datos
        async function saveState() {
            try {
                const response = await fetch('../../api/actualizar_estado_ruta.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        curso: '{$curso}',
                        unidad: '{$unidad}',
                        actividad: '{$actividad}',
                        estrellas: gameState.stars,
                        puntos: gameState.totalPoints,
                        nivel: gameState.currentLevel,
                        intentos: gameState.totalAttempts
                    })
                });

                const result = await response.json();
                if (result.success) {
                    console.log('✅ Estado guardado en BD exitosamente');
                } else {
                    console.error('❌ Error al guardar en BD:', result.message);
                }
            } catch (error) {
                console.error('❌ Error de conexión al guardar:', error);
            }
        }

        // Actualizar interfaz
        function updateUI() {
            const currentLevelConfig = levels[gameState.currentLevel];
            const progress = Math.min((gameState.totalPoints / currentLevelConfig.requiredPoints) * 100, 100);
            
            // Actualizar estrellas
            for (let i = 1; i <= 3; i++) {
                const star = document.getElementById('star-' + i);
                if (i <= gameState.stars) {
                    star.classList.add('earned');
                } else {
                    star.classList.remove('earned');
                }
            }

            // Actualizar título del nivel actual
            const currentLevelTitle = document.getElementById('current-level-title');
            if (currentLevelTitle) {
                if (gameState.stars === 0) {
                    currentLevelTitle.textContent = 'Nivel Inicial';
                } else if (gameState.stars === 1) {
                    currentLevelTitle.textContent = 'Nivel Intermedio';
                } else if (gameState.stars === 2) {
                    currentLevelTitle.textContent = 'Nivel Avanzado';
                } else if (gameState.stars >= 3) {
                    currentLevelTitle.textContent = 'Nivel Experto';
                }
            }

            // Actualizar barra de progreso
            const progressBar = document.getElementById('progress-bar');
            const progressText = document.getElementById('progress-text');
            if (progressBar) {
                progressBar.style.width = progress + '%';
            }
            if (progressText) {
                progressText.textContent = Math.round(progress) + '%';
            }

            // Actualizar mensaje según las estrellas
            const levelMessage = document.getElementById('level-message');
            if (levelMessage) {
                if (gameState.stars === 0) {
                    levelMessage.textContent = 'Vamos rumbo al Nivel Intermedio';
                } else if (gameState.stars === 1) {
                    levelMessage.textContent = 'Vamos rumbo al Nivel Avanzado';
                } else if (gameState.stars === 2) {
                    levelMessage.textContent = 'Vamos rumbo al Nivel Experto';
                } else if (gameState.stars >= 3) {
                    levelMessage.textContent = '¡Campeón del Nivel Experto!';
                }
            }
        }

        // Cargar ejercicio aleatorio
        async function loadRandomExercise() {
            const currentLevelConfig = levels[gameState.currentLevel];
            const itemFile = await getRandomItem();
            const iframe = document.getElementById('exercise-iframe');
            
            console.log('🔍 Cargando ejercicio aleatorio...');
            console.log('- Nivel actual:', gameState.currentLevel);
            console.log('- Carpeta:', currentLevelConfig.folder);
            console.log('- Item seleccionado:', itemFile);
            
            if (currentLevelConfig && itemFile) {
                const fullPath = currentLevelConfig.folder + '/' + itemFile;
                iframe.src = fullPath;
                
                // Establecer el item actual
                currentItem = parseInt(itemFile.replace('item', '').replace('.html', ''));
                
                // Reiniciar el contador de intentos para el nuevo item
                attemptNumber = 0;
                
                console.log('✅ Cargando ejercicio: ' + fullPath);
                
                // Esperar a que el iframe cargue para obtener los puntos
                iframe.onload = function() {
                    // Intentar obtener los puntos del iframe
                    try {
                        const iframeDoc = iframe.contentDocument || iframe.contentWindow.document;
                        const submitBtn = iframeDoc.querySelector('.btn-submit');
                        if (submitBtn) {
                            const onclickAttr = submitBtn.getAttribute('onclick');
                            if (onclickAttr) {
                                // Extraer los puntos del onclick="checkAnswer('a', 10)"
                                const match = onclickAttr.match(/checkAnswer\([^,]+,\s*(\d+)\)/);
                                if (match) {
                                    currentItemPoints = parseInt(match[1]);
                                }
                            }
                        }
                        
                        // Enviar los puntos al iframe para que actualice el título
                        if (iframe.contentWindow && currentItemPoints > 0) {
                            iframe.contentWindow.postMessage({
                                type: 'updatePoints',
                                points: currentItemPoints
                            }, '*');
                        }
                    } catch (e) {
                        console.log('No se pudieron obtener los puntos del iframe');
                        currentItemPoints = 5; // Valor por defecto
                    }
                    
                    // Mostrar información del ejercicio
                    mostrarInfoEjercicio();
                };
            } else {
                console.error('❌ Error: No se pudo cargar el ejercicio');
                iframe.src = '';
            }
        }
        
        // Función para mostrar información temporal del ejercicio
        function mostrarInfoEjercicio() {
            const infoDiv = document.getElementById('exercise-info');
            const infoPoints = document.getElementById('info-points');
            const infoItem = document.getElementById('info-item');
            const infoLevel = document.getElementById('info-level');
            const infoAttempt = document.getElementById('info-attempt');
            const progressNivel1 = document.getElementById('progress-nivel1');
            const progressNivel2 = document.getElementById('progress-nivel2');
            const progressNivel3 = document.getElementById('progress-nivel3');
            
            // Determinar nombre del nivel basado en estrellas (numeric)
            const starsNum = parseInt(gameState.stars, 10) || 0;
            let nombreNivel = '';
            if (starsNum === 0) nombreNivel = 'Inicial';
            else if (starsNum === 1) nombreNivel = 'Intermedio';
            else if (starsNum === 2) nombreNivel = 'Avanzado';
            else nombreNivel = 'Experto';
            
            // Calcular progreso para cada nivel
            const totalPoints = parseInt(gameState.totalPoints, 10) || 0;
            
            // Nivel 1: 0 a 20 puntos
            const puntosNivel1 = Math.min(totalPoints, levels[1].requiredPoints);
            const maxNivel1 = levels[1].requiredPoints;
            
            // Nivel 2: necesita los puntos del nivel 1 completos + su progreso
            const puntosNivel2 = totalPoints < levels[1].requiredPoints ? 0 : 
                                Math.min(totalPoints - levels[1].requiredPoints, levels[2].requiredPoints - levels[1].requiredPoints);
            const maxNivel2 = levels[2].requiredPoints - levels[1].requiredPoints;
            
            // Nivel 3: necesita los puntos del nivel 2 completos + su progreso
            const puntosNivel3 = totalPoints < levels[2].requiredPoints ? 0 : 
                                Math.min(totalPoints - levels[2].requiredPoints, levels[3].requiredPoints - levels[2].requiredPoints);
            const maxNivel3 = levels[3].requiredPoints - levels[2].requiredPoints;
            
            // Actualizar información
            infoPoints.innerHTML = `Puntaje Acumulado: <strong>${gameState.totalPoints}</strong>`;
            infoItem.innerHTML = `Item: <strong>item${String(currentItem).padStart(2, '0')}.html</strong>`;
            infoLevel.innerHTML = `Nivel: <strong>${nombreNivel}</strong> (${starsNum})`;
            infoAttempt.innerHTML = `Intento: <strong>${gameState.totalAttempts}</strong>`;
            
            // Actualizar progreso de niveles
            progressNivel1.innerHTML = `N1: ${puntosNivel1}/${maxNivel1}`;
            progressNivel2.innerHTML = `N2: ${puntosNivel2}/${maxNivel2}`;
            progressNivel3.innerHTML = `N3: ${puntosNivel3}/${maxNivel3}`;
            
            // Mostrar el mensaje
            infoDiv.style.display = 'block';
        }

        // Obtener lista de items disponibles en el nivel actual
        async function getAvailableItemsInLevel() {
            const currentLevelConfig = levels[gameState.currentLevel];
            const folder = currentLevelConfig.folder;
            
            try {
                // Intentar cargar items del 1 al 20 y ver cuáles existen
                const itemsExistentes = [];
                const maxPossibleItems = 20; // Límite razonable para verificar
                
                for (let i = 1; i <= maxPossibleItems; i++) {
                    const itemFile = 'item' + String(i).padStart(2, '0') + '.html';
                    const testUrl = folder + '/' + itemFile;
                    
                    try {
                        const response = await fetch(testUrl, { method: 'HEAD' });
                        if (response.ok) {
                            itemsExistentes.push(i);
                        }
                    } catch (e) {
                        // El archivo no existe, continuar
                    }
                }
                
                return itemsExistentes;
            } catch (error) {
                console.error('Error al obtener items disponibles:', error);
                return [1]; // Fallback: asumir que al menos existe item01
            }
        }

        // Obtener item aleatorio
        async function getRandomItem() {
            const itemsDisponibles = await getAvailableItemsInLevel();
            
            if (itemsDisponibles.length === 0) {
                console.error('❌ No hay items disponibles en este nivel');
                return null;
            }
            
            // Filtrar items que no han sido usados
            let availableItems = itemsDisponibles.filter(item => !gameState.usedItems.includes(item));
            
            // Si ya usamos todos, reiniciar la lista
            if (availableItems.length === 0) {
                gameState.usedItems = [];
                availableItems = itemsDisponibles;
            }
            
            // Seleccionar uno aleatorio
            const randomIndex = Math.floor(Math.random() * availableItems.length);
            const itemNumber = availableItems[randomIndex];
            const result = 'item' + String(itemNumber).padStart(2, '0') + '.html';
            
            console.log('✅ Items disponibles en nivel:', itemsDisponibles);
            console.log('✅ Item seleccionado:', result);
            
            return result;
        }

        // Verificar si se completó el nivel
        function checkLevelComplete() {
            const currentLevelConfig = levels[gameState.currentLevel];
            
            if (gameState.totalPoints >= currentLevelConfig.requiredPoints) {
                if (gameState.stars < 3) {
                    gameState.stars++;
                    gameState.currentLevel = Math.min(gameState.currentLevel + 1, 3);
                    
                    // Mostrar mensaje de nivel completado
                    const levelNames = ['Inicial', 'Intermedio', 'Avanzado', 'Experto'];
                    alert('¡Nivel ' + levelNames[gameState.stars - 1] + ' Completado! ¡Felicidades!');
                    
                    updateUI();
                    saveState();
                }
            }
            
            // Cargar un nuevo ejercicio aleatorio
            loadRandomExercise();
        }

        // Inicializar
        async function init() {
            await loadState();
            updateUI();
            loadRandomExercise();
        }

        // Escuchar mensajes del iframe
        window.addEventListener('message', function(event) {
            if (event.data && event.data.type === 'answer') {
                // Incrementar el contador de intentos
                attemptNumber++;
                gameState.totalAttempts++;
                
                if (event.data.correct) {
                    gameState.totalPoints += parseInt(event.data.points) || 0;
                    gameState.usedItems.push(currentItem); // Agregar item usado
                    
                    updateUI();
                    saveState(); // ¡IMPORTANTE! Guardar en BD
                    
                    // Llamar a checkLevelComplete para verificar si se completó el nivel
                    checkLevelComplete();
                } else {
                    // Respuesta incorrecta - solo incrementar intentos
                    gameState.usedItems.push(currentItem); // Agregar item usado
                    
                    updateUI();
                    saveState(); // Guardar intentos en BD
                    
                    // Llamar a checkLevelComplete para cargar el siguiente ejercicio
                    checkLevelComplete();
                }
            }
        });

        // Inicializar cuando se carga la página
        init();
    </script>
</body>
</html>";

    return $contenido;
}

try {
    // Verificar que sea una petición POST
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        throw new Exception('Método no permitido');
    }

    // Obtener datos del JSON
    $input = file_get_contents('php://input');
    $data = json_decode($input, true);

    if (!$data) {
        throw new Exception('Datos JSON inválidos');
    }

    $ruta_nombre = $data['ruta_nombre'] ?? '';
    $niveles = $data['niveles'] ?? [];

    if (empty($ruta_nombre) || empty($niveles)) {
        throw new Exception('Datos incompletos');
    }

    // Crear archivo de configuración JSON
    $config_path = "../rutas/{$ruta_nombre}/config.json";
    
    // Crear directorio si no existe
    $dir = dirname($config_path);
    if (!is_dir($dir)) {
        mkdir($dir, 0755, true);
    }

    // Preparar datos de configuración
    $config_data = [
        'ruta_nombre' => $ruta_nombre,
        'niveles' => $niveles,
        'fecha_actualizacion' => date('Y-m-d H:i:s'),
        'version' => '1.0'
    ];

    // Guardar configuración
    $json_data = json_encode($config_data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    
    if (file_put_contents($config_path, $json_data) === false) {
        throw new Exception('Error al guardar el archivo de configuración');
    }

    // Crear carpetas de niveles
    $carpetas_creadas = [];
    foreach ($niveles as $nivel) {
        $carpeta_nivel = $dir . '/' . $nivel['carpeta'];
        if (!is_dir($carpeta_nivel)) {
            if (mkdir($carpeta_nivel, 0755, true)) {
                $carpetas_creadas[] = $nivel['carpeta'];
                error_log("Carpeta creada: {$carpeta_nivel}");
            } else {
                error_log("Error al crear carpeta: {$carpeta_nivel}");
            }
        } else {
            $carpetas_creadas[] = $nivel['carpeta'] . ' (ya existía)';
        }
    }

    // Crear archivo principal de la ruta
    $archivo_ruta = $dir . '/' . $ruta_nombre . '.php';
    $contenido_ruta = generarArchivoRuta($ruta_nombre, $niveles);
    
    if (file_put_contents($archivo_ruta, $contenido_ruta) === false) {
        error_log("Error al crear archivo principal de la ruta: {$archivo_ruta}");
    } else {
        error_log("Archivo principal de la ruta creado: {$archivo_ruta}");
    }

    // Copiar archivos necesarios desde la ruta de referencia
    $archivos_copiados = [];
    
    // Copiar item.css
    $item_css_origen = "../rutas/ruta_c001_u01_act03/item.css";
    $item_css_destino = $dir . "/item.css";
    
    if (file_exists($item_css_origen)) {
        if (copy($item_css_origen, $item_css_destino)) {
            $archivos_copiados[] = "item.css";
            error_log("Archivo item.css copiado exitosamente");
        } else {
            error_log("Error al copiar item.css");
        }
    } else {
        error_log("Archivo item.css de referencia no encontrado: {$item_css_origen}");
    }
    
    // Copiar item.js
    $item_js_origen = "../rutas/ruta_c001_u01_act03/item.js";
    $item_js_destino = $dir . "/item.js";
    
    if (file_exists($item_js_origen)) {
        if (copy($item_js_origen, $item_js_destino)) {
            $archivos_copiados[] = "item.js";
            error_log("Archivo item.js copiado exitosamente");
        } else {
            error_log("Error al copiar item.js");
        }
    } else {
        error_log("Archivo item.js de referencia no encontrado: {$item_js_origen}");
    }

    // Log de la operación
    error_log("Configuración de niveles guardada para ruta: {$ruta_nombre}");
    error_log("Carpetas de niveles: " . implode(', ', $carpetas_creadas));

    echo json_encode([
        'success' => true,
        'message' => 'Configuración de niveles guardada exitosamente',
        'ruta_nombre' => $ruta_nombre,
        'niveles_guardados' => count($niveles),
        'carpetas_creadas' => $carpetas_creadas,
        'archivo_config' => $config_path,
        'archivo_ruta' => $archivo_ruta,
        'archivos_copiados' => $archivos_copiados,
        'item_css_copiado' => file_exists($item_css_destino),
        'item_js_copiado' => file_exists($item_js_destino)
    ]);

} catch (Exception $e) {
    error_log("Error en guardar_configuracion_niveles.php: " . $e->getMessage());
    
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}
?>
