<?php
session_start();
header('Content-Type: application/json');

// Habilitar logging de errores para debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('log_errors', 1);

// Incluir configuración de la base de datos
require_once dirname(__FILE__) . '/../config.php';
require_once dirname(__FILE__) . '/../conexion.php';

$response = ['success' => false, 'message' => '', 'details' => []];

// Log para debugging
error_log("API guardar_respuesta_larga.php - Iniciando procesamiento");

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $input = json_decode(file_get_contents('php://input'), true);
    
    // Log de datos recibidos
    error_log("Datos recibidos: " . json_encode($input));

    $tipo = $input['tipo'] ?? null;
    $numero_respuesta = $input['numero_respuesta'] ?? null;
    $curso = $input['curso'] ?? null;
    $unidad = $input['unidad'] ?? null;
    $actividad = $input['actividad'] ?? null;
    $contenido = $input['contenido'] ?? null;
    $palabras = $input['palabras'] ?? null;

    // Validaciones básicas
    if (!$tipo || !$numero_respuesta || !$curso || !$unidad || !$actividad) {
        $response['message'] = 'Parámetros incompletos.';
        error_log("Error: Parámetros incompletos - tipo: $tipo, numero: $numero_respuesta, curso: $curso, unidad: $unidad, actividad: $actividad");
        echo json_encode($response);
        exit;
    }
    
    error_log("Parámetros validados correctamente");

    // Normalizar formato de unidad (Unidad 01 -> 01)
    $unidad_normalizada = '';
    if (preg_match('/Unidad (\d+)/', $unidad, $matches)) {
        $unidad_normalizada = str_pad($matches[1], 2, '0', STR_PAD_LEFT);
    }

    // Asegurarse de que la actividad sea un número de dos dígitos
    $actividad_padded = str_pad($actividad, 2, '0', STR_PAD_LEFT);
    $numero_respuesta_padded = str_pad($numero_respuesta, 2, '0', STR_PAD_LEFT);

    $table_name = "avance_{$curso}_u{$unidad_normalizada}";
    
    // Determinar el nombre de la columna según el tipo
    $column_name = '';
    $fecha_column = '';
    $palabras_column = '';
    $enviado_column = '';
    
    switch ($tipo) {
        case 'respuesta':
            $column_name = "RL{$numero_respuesta_padded}ACT{$actividad_padded}";
            $fecha_column = "FRL{$numero_respuesta_padded}ACT{$actividad_padded}";
            $palabras_column = "PRL{$numero_respuesta_padded}ACT{$actividad_padded}";
            $enviado_column = "RLE{$numero_respuesta_padded}ACT{$actividad_padded}";
            break;
        case 'retroalimentacion':
            $column_name = "RTRL{$numero_respuesta_padded}ACT{$actividad_padded}";
            $fecha_column = "FRTRL{$numero_respuesta_padded}ACT{$actividad_padded}";
            break;
        case 'nota':
            $column_name = "NTRL{$numero_respuesta_padded}ACT{$actividad_padded}";
            $fecha_column = "FNTRL{$numero_respuesta_padded}ACT{$actividad_padded}";
            break;
        case 'eliminar_respuesta':
            $column_name = "RL{$numero_respuesta_padded}ACT{$actividad_padded}";
            $fecha_column = "FRL{$numero_respuesta_padded}ACT{$actividad_padded}";
            $palabras_column = "PRL{$numero_respuesta_padded}ACT{$actividad_padded}";
            $enviado_column = "RLE{$numero_respuesta_padded}ACT{$actividad_padded}";
            break;
        case 'eliminar_retroalimentacion':
            $column_name = "RTRL{$numero_respuesta_padded}ACT{$actividad_padded}";
            $fecha_column = "FRTRL{$numero_respuesta_padded}ACT{$actividad_padded}";
            break;
        case 'fecha_limite':
            $column_name = "FLRL{$numero_respuesta_padded}ACT{$actividad_padded}";
            break;
        default:
            $response['message'] = 'Tipo de dato no válido.';
            echo json_encode($response);
            exit;
    }

    try {
        error_log("Verificando existencia de tabla: $table_name");
        
        // 1. Verificar si la tabla existe
        $stmt = $dsn->prepare("SHOW TABLES LIKE :table_name");
        $stmt->execute([':table_name' => $table_name]);
        $table_exists = $stmt->fetch() !== false;
        
        error_log("Tabla existe: " . ($table_exists ? 'SÍ' : 'NO'));

        if (!$table_exists) {
            $response['message'] = "La tabla '{$table_name}' no existe. Primero debe crearse usando el botón 'Crear tabla' en el panel.";
            error_log("Error: Tabla no existe - $table_name");
            echo json_encode($response);
            exit;
        }

        // 2. Verificar si la columna existe
        $stmt_check_col = $dsn->prepare("SHOW COLUMNS FROM `{$table_name}` LIKE :col_name");
        $stmt_check_col->execute([':col_name' => $column_name]);
        $col_exists = $stmt_check_col->fetch() !== false;

        if (!$col_exists) {
            // Determinar el tipo de columna según el tipo de dato
            $column_type = '';
            switch ($tipo) {
                case 'respuesta':
                case 'retroalimentacion':
                    $column_type = "TEXT NULL DEFAULT NULL";
                    break;
                case 'nota':
                    $column_type = "DECIMAL(5,2) NULL DEFAULT NULL";
                    break;
            }
            
            // Agregar la columna
            $add_column_sql = "ALTER TABLE `{$table_name}` ADD COLUMN `{$column_name}` {$column_type}";
            $dsn->exec($add_column_sql);
            $response['details'][] = "Columna '{$column_name}' agregada a '{$table_name}'.";
        }

        // 3. Obtener el ID del usuario actual (asumiendo que está en la sesión)
        $user_id = $_SESSION['user_id'] ?? null;
        if (!$user_id) {
            // Si no hay usuario en sesión, usar un ID por defecto para pruebas
            // En producción, esto debería manejarse con autenticación adecuada
            $user_id = 1; // ID por defecto para pruebas
            $response['details'][] = "Usando ID de usuario por defecto: {$user_id}";
        }

        // 4. Verificar si el usuario existe en la tabla
        $stmt_check_user = $dsn->prepare("SELECT id FROM `{$table_name}` WHERE id = :user_id");
        $stmt_check_user->execute([':user_id' => $user_id]);
        $user_exists = $stmt_check_user->fetch() !== false;

        if (!$user_exists) {
            // Insertar el usuario con valores NULL para todas las columnas
            $insert_user_sql = "INSERT INTO `{$table_name}` (`id`) VALUES (:user_id)";
            $stmt_insert_user = $dsn->prepare($insert_user_sql);
            $stmt_insert_user->execute([':user_id' => $user_id]);
            $response['details'][] = "Usuario {$user_id} agregado a '{$table_name}'.";
        }

        // 5. Actualizar el valor en la base de datos
        $fecha_actual = date('Y-m-d H:i:s');
        $update_sql = "UPDATE `{$table_name}` SET `{$column_name}` = :contenido";
        $params = [':contenido' => null, ':user_id' => $user_id];
        
        // Agregar fecha si corresponde
        if ($fecha_column) {
            $update_sql .= ", `{$fecha_column}` = :fecha";
            $params[':fecha'] = $fecha_actual;
        }
        
        // Agregar palabras si corresponde
        if ($palabras_column && $palabras !== null) {
            $update_sql .= ", `{$palabras_column}` = :palabras";
            $params[':palabras'] = intval($palabras);
        }
        
        // Agregar variable de enviado si corresponde
        if ($enviado_column) {
            if ($tipo === 'eliminar_respuesta') {
                // Si se elimina la respuesta, poner RLE a NULL
                $update_sql .= ", `{$enviado_column}` = NULL";
            } elseif ($tipo === 'respuesta' && $contenido) {
                // Si se guarda una respuesta, poner RLE a 1
                $update_sql .= ", `{$enviado_column}` = 1";
            }
        }
        
        $update_sql .= " WHERE id = :user_id";
        
        // Preparar el contenido según el tipo
        if ($tipo === 'eliminar_respuesta' || $tipo === 'eliminar_retroalimentacion') {
            $params[':contenido'] = null;
            if ($fecha_column) {
                $params[':fecha'] = null;
            }
        } elseif ($tipo === 'fecha_limite') {
            $params[':contenido'] = $contenido; // Usar la fecha límite del formulario
        } else {
            $contenido_preparado = $contenido;
            if ($tipo === 'nota') {
                $contenido_preparado = floatval($contenido);
            }
            $params[':contenido'] = $contenido_preparado;
        }
        
        $stmt_update = $dsn->prepare($update_sql);
        $stmt_update->execute($params);

        $response['success'] = true;
        $response['message'] = "Datos guardados exitosamente en '{$column_name}'.";
        $response['details'][] = "Usuario: {$user_id}";
        $response['details'][] = "Tabla: {$table_name}";
        $response['details'][] = "Columna: {$column_name}";
        $response['details'][] = "Tipo: {$tipo}";

    } catch (PDOException $e) {
        $response['message'] = "Error de base de datos: " . $e->getMessage();
        error_log("Error en guardar_respuesta_larga.php: " . $e->getMessage());
    } catch (Exception $e) {
        $response['message'] = "Error inesperado: " . $e->getMessage();
        error_log("Error inesperado en guardar_respuesta_larga.php: " . $e->getMessage());
    }
} else {
    $response['message'] = 'Método no permitido.';
}

echo json_encode($response);
?>
