<?php
/**
 * Asigna un rol completo a todos los usuarios de un curso específico
 * Actualiza masivamente la columna del curso seleccionado
 */

// Configurar headers para JSON
header('Content-Type: application/json; charset=utf-8');
header('Cache-Control: no-cache, no-store, must-revalidate');
header('Pragma: no-cache');
header('Expires: 0');

// Verificar que sea una petición POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Método no permitido']);
    exit;
}

// Obtener el contenido JSON
$input = file_get_contents('php://input');
$data = json_decode($input, true);

if (!$data) {
    http_response_code(400);
    echo json_encode(['error' => 'Datos JSON inválidos']);
    exit;
}

// Extraer datos
$dbHost = $data['dbHost'] ?? '';
$dbName = $data['dbName'] ?? '';
$dbUser = $data['dbUser'] ?? '';
$dbPassword = $data['dbPassword'] ?? '';
$dbPort = $data['dbPort'] ?? '3306';
$nombreTabla = $data['nombreTabla'] ?? '';
$securityKey = $data['securityKey'] ?? '';
$curso = $data['curso'] ?? '';
$rol = $data['rol'] ?? null;

// Validar datos requeridos
if (empty($dbHost) || empty($dbName) || empty($dbUser) || empty($dbPort) || empty($nombreTabla)) {
    http_response_code(400);
    echo json_encode(['error' => 'Faltan datos requeridos']);
    exit;
}

// Validar curso
if (empty($curso) || !preg_match('/^c\d{3}$/', $curso)) {
    http_response_code(400);
    echo json_encode(['error' => 'Curso inválido']);
    exit;
}

// Validar rol (puede ser null o un número del 0 al 4)
if ($rol !== null && (!is_numeric($rol) || $rol < 0 || $rol > 4)) {
    http_response_code(400);
    echo json_encode(['error' => 'Rol inválido']);
    exit;
}

// Validar clave de seguridad
$expectedKey = 'clave_secreta_123'; // Cambiar por una clave más segura en producción
if ($securityKey !== $expectedKey) {
    http_response_code(403);
    echo json_encode(['error' => 'Clave de seguridad incorrecta']);
    exit;
}

// Función para registrar logs
function logAsignarRol($message) {
    $logFile = '../logs/asignar_rol_completo_inscripcion_cursos.log';
    $timestamp = date('Y-m-d H:i:s');
    $logMessage = "[$timestamp] $message\n";
    
    // Crear directorio de logs si no existe
    if (!is_dir('../logs')) {
        mkdir('../logs', 0755, true);
    }
    
    file_put_contents($logFile, $logMessage, FILE_APPEND | LOCK_EX);
}

try {
    // Registrar intento de asignación
    $rolTexto = $rol === null ? 'NULL' : $rol;
    logAsignarRol("Asignación de rol - Host: $dbHost, DB: $dbName, Tabla: $nombreTabla, Curso: $curso, Rol: $rolTexto");
    
    // Crear conexión PDO
    $dsn = "mysql:host=$dbHost;port=$dbPort;dbname=$dbName;charset=utf8mb4";
    $options = [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        PDO::ATTR_EMULATE_PREPARES => false,
    ];
    
    $pdo = new PDO($dsn, $dbUser, $dbPassword, $options);
    
    // Verificar si la tabla existe
    $stmt = $pdo->prepare("SELECT COUNT(*) as existe FROM information_schema.tables WHERE table_schema = ? AND table_name = ?");
    $stmt->execute([$dbName, $nombreTabla]);
    $result = $stmt->fetch();
    
    if ($result['existe'] == 0) {
        logAsignarRol("Error: Tabla no existe - $nombreTabla");
        
        echo json_encode([
            'success' => false,
            'message' => 'La tabla no existe en la base de datos',
            'timestamp' => date('Y-m-d H:i:s')
        ]);
        exit;
    }
    
    // Verificar si la columna del curso existe
    $stmt = $pdo->prepare("SELECT COUNT(*) as existe FROM information_schema.columns WHERE table_schema = ? AND table_name = ? AND column_name = ?");
    $stmt->execute([$dbName, $nombreTabla, $curso]);
    $result = $stmt->fetch();
    
    if ($result['existe'] == 0) {
        logAsignarRol("Error: Columna del curso no existe - $curso");
        
        echo json_encode([
            'success' => false,
            'message' => "La columna del curso $curso no existe en la tabla",
            'timestamp' => date('Y-m-d H:i:s')
        ]);
        exit;
    }
    
    // Obtener el total de registros antes de la actualización
    $stmt = $pdo->prepare("SELECT COUNT(*) as total FROM `$nombreTabla`");
    $stmt->execute();
    $totalRegistros = $stmt->fetch()['total'];
    
    if ($totalRegistros == 0) {
        logAsignarRol("Advertencia: No hay registros en la tabla para actualizar");
        
        echo json_encode([
            'success' => false,
            'message' => 'No hay registros en la tabla para actualizar',
            'timestamp' => date('Y-m-d H:i:s')
        ]);
        exit;
    }
    
    // Comenzar transacción
    $pdo->beginTransaction();
    
    // Preparar la consulta de actualización
    if ($rol === null) {
        $sql = "UPDATE `$nombreTabla` SET `$curso` = NULL";
        $params = [];
    } else {
        $sql = "UPDATE `$nombreTabla` SET `$curso` = ?";
        $params = [$rol];
    }
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    
    $registrosActualizados = $stmt->rowCount();
    
    // Confirmar transacción
    $pdo->commit();
    
    logAsignarRol("Asignación de rol exitosa - Curso: $curso, Rol: $rolTexto, Registros actualizados: $registrosActualizados");
    
    // Retornar resultado exitoso
    echo json_encode([
        'success' => true,
        'message' => "Rol asignado exitosamente al curso $curso",
        'curso' => $curso,
        'rol' => $rol,
        'totalRegistros' => $totalRegistros,
        'registrosActualizados' => $registrosActualizados,
        'timestamp' => date('Y-m-d H:i:s')
    ]);
    
    // Cerrar conexión
    $pdo = null;
    
} catch (PDOException $e) {
    // Error de base de datos
    if (isset($pdo) && $pdo->inTransaction()) {
        $pdo->rollBack();
    }
    
    logAsignarRol("Error de base de datos: " . $e->getMessage());
    
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Error de base de datos',
        'message' => $e->getMessage(),
        'timestamp' => date('Y-m-d H:i:s')
    ]);
    
} catch (Exception $e) {
    // Error general
    if (isset($pdo) && $pdo->inTransaction()) {
        $pdo->rollBack();
    }
    
    logAsignarRol("Error general: " . $e->getMessage());
    
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Error interno del servidor',
        'message' => $e->getMessage(),
        'timestamp' => date('Y-m-d H:i:s')
    ]);
}
?> 