<?php
/**
 * Creador de tabla de inscripción de cursos - Versión Debug
 * Crea la tabla con la estructura correcta para roles de usuario por curso
 */

// Habilitar reporte de errores para debug
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Configurar headers para JSON
header('Content-Type: application/json; charset=utf-8');
header('Cache-Control: no-cache, no-store, must-revalidate');
header('Pragma: no-cache');
header('Expires: 0');

// Función para registrar logs
function logCreacion($message) {
    $logFile = '../logs/creacion_tabla_inscripcion_cursos.log';
    $timestamp = date('Y-m-d H:i:s');
    $logMessage = "[$timestamp] $message\n";
    
    // Crear directorio de logs si no existe
    if (!is_dir('../logs')) {
        mkdir('../logs', 0755, true);
    }
    
    file_put_contents($logFile, $logMessage, FILE_APPEND | LOCK_EX);
}

try {
    logCreacion("=== INICIO DE CREACIÓN DE TABLA ===");
    
    // Verificar método HTTP
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        logCreacion("Error: Método no permitido - " . $_SERVER['REQUEST_METHOD']);
        http_response_code(405);
        echo json_encode(['error' => 'Método no permitido']);
        exit;
    }
    
    logCreacion("Método HTTP válido: POST");
    
    // Obtener el contenido JSON
    $input = file_get_contents('php://input');
    logCreacion("Input recibido: " . substr($input, 0, 200) . "...");
    
    $data = json_decode($input, true);
    
    if (!$data) {
        logCreacion("Error: Datos JSON inválidos");
        http_response_code(400);
        echo json_encode(['error' => 'Datos JSON inválidos']);
        exit;
    }
    
    logCreacion("JSON decodificado correctamente");
    
    // Extraer datos
    $dbHost = $data['dbHost'] ?? '';
    $dbName = $data['dbName'] ?? '';
    $dbUser = $data['dbUser'] ?? '';
    $dbPassword = $data['dbPassword'] ?? '';
    $dbPort = $data['dbPort'] ?? '3306';
    $nombreTabla = $data['nombreTabla'] ?? '';
    $securityKey = $data['securityKey'] ?? '';
    
    logCreacion("Datos extraídos - Host: $dbHost, DB: $dbName, Tabla: $nombreTabla, Usuario: $dbUser, Puerto: $dbPort");
    
    // Validar datos requeridos
    if (empty($dbHost) || empty($dbName) || empty($dbUser) || empty($dbPort) || empty($nombreTabla)) {
        logCreacion("Error: Faltan datos requeridos");
        http_response_code(400);
        echo json_encode(['error' => 'Faltan datos requeridos']);
        exit;
    }
    
    logCreacion("Validación de datos requeridos exitosa");
    
    // Validar clave de seguridad
    $expectedKey = 'clave_secreta_123';
    if ($securityKey !== $expectedKey) {
        logCreacion("Error: Clave de seguridad incorrecta");
        http_response_code(403);
        echo json_encode(['error' => 'Clave de seguridad incorrecta']);
        exit;
    }
    
    logCreacion("Clave de seguridad válida");
    
    // Crear conexión PDO
    logCreacion("Intentando conexión PDO...");
    
    $dsn = "mysql:host=$dbHost;port=$dbPort;dbname=$dbName;charset=utf8mb4";
    logCreacion("DSN: $dsn");
    
    $options = [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        PDO::ATTR_EMULATE_PREPARES => false,
    ];
    
    $pdo = new PDO($dsn, $dbUser, $dbPassword, $options);
    logCreacion("Conexión PDO exitosa");
    
    // Verificar si la tabla ya existe
    logCreacion("Verificando si la tabla ya existe...");
    
    $stmt = $pdo->prepare("SELECT COUNT(*) as existe FROM information_schema.tables WHERE table_schema = ? AND table_name = ?");
    $stmt->execute([$dbName, $nombreTabla]);
    $result = $stmt->fetch();
    
    logCreacion("Resultado verificación: " . json_encode($result));
    
    if ($result['existe'] > 0) {
        logCreacion("Error: Tabla ya existe - $nombreTabla");
        
        echo json_encode([
            'success' => false,
            'message' => 'La tabla ya existe en la base de datos',
            'timestamp' => date('Y-m-d H:i:s')
        ]);
        exit;
    }
    
    logCreacion("Tabla no existe, procediendo a crear...");
    
    // Crear la tabla con estructura simplificada (solo 10 cursos)
    $sql = "CREATE TABLE `$nombreTabla` (
        `id` int(11) NOT NULL COMMENT 'ID fijo del usuario (no auto-increment)',
        `c001` tinyint(1) NULL DEFAULT NULL COMMENT 'Rol: NULL=No asignado, 0=Estudiante, 1=Profesor, 2=Revisor, 3=Observador, 4=Administrador',
        `c002` tinyint(1) NULL DEFAULT NULL COMMENT 'Rol: NULL=No asignado, 0=Estudiante, 1=Profesor, 2=Revisor, 3=Observador, 4=Administrador',
        `c003` tinyint(1) NULL DEFAULT NULL COMMENT 'Rol: NULL=No asignado, 0=Estudiante, 1=Profesor, 2=Revisor, 3=Observador, 4=Administrador',
        `c004` tinyint(1) NULL DEFAULT NULL COMMENT 'Rol: NULL=No asignado, 0=Estudiante, 1=Profesor, 2=Revisor, 3=Observador, 4=Administrador',
        `c005` tinyint(1) NULL DEFAULT NULL COMMENT 'Rol: NULL=No asignado, 0=Estudiante, 1=Profesor, 2=Revisor, 3=Observador, 4=Administrador',
        `c006` tinyint(1) NULL DEFAULT NULL COMMENT 'Rol: NULL=No asignado, 0=Estudiante, 1=Profesor, 2=Revisor, 3=Observador, 4=Administrador',
        `c007` tinyint(1) NULL DEFAULT NULL COMMENT 'Rol: NULL=No asignado, 0=Estudiante, 1=Profesor, 2=Revisor, 3=Observador, 4=Administrador',
        `c008` tinyint(1) NULL DEFAULT NULL COMMENT 'Rol: NULL=No asignado, 0=Estudiante, 1=Profesor, 2=Revisor, 3=Observador, 4=Administrador',
        `c009` tinyint(1) NULL DEFAULT NULL COMMENT 'Rol: NULL=No asignado, 0=Estudiante, 1=Profesor, 2=Revisor, 3=Observador, 4=Administrador',
        `c010` tinyint(1) NULL DEFAULT NULL COMMENT 'Rol: NULL=No asignado, 0=Estudiante, 1=Profesor, 2=Revisor, 3=Observador, 4=Administrador',
        PRIMARY KEY (`id`)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    
    logCreacion("SQL generado: " . $sql);
    
    // Ejecutar la creación de la tabla
    logCreacion("Ejecutando SQL...");
    $pdo->exec($sql);
    logCreacion("SQL ejecutado exitosamente");
    
    // Verificar que la tabla se creó correctamente
    logCreacion("Verificando creación de tabla...");
    
    $stmt = $pdo->prepare("SELECT COUNT(*) as existe FROM information_schema.tables WHERE table_schema = ? AND table_name = ?");
    $stmt->execute([$dbName, $nombreTabla]);
    $result = $stmt->fetch();
    
    logCreacion("Verificación post-creación: " . json_encode($result));
    
    if ($result['existe'] > 0) {
        logCreacion("Tabla creada exitosamente - $nombreTabla");
        
        // Obtener información detallada de la tabla creada
        $stmt = $pdo->prepare("
            SELECT COUNT(*) as total_columnas
            FROM information_schema.columns 
            WHERE table_schema = ? AND table_name = ?
        ");
        $stmt->execute([$dbName, $nombreTabla]);
        $columnas = $stmt->fetch();
        
        // Obtener información de la estructura
        $stmt = $pdo->prepare("
            SELECT COLUMN_NAME, DATA_TYPE, IS_NULLABLE, COLUMN_DEFAULT, COLUMN_COMMENT
            FROM information_schema.columns 
            WHERE table_schema = ? AND table_name = ?
            ORDER BY ORDINAL_POSITION
        ");
        $stmt->execute([$dbName, $nombreTabla]);
        $estructura = $stmt->fetchAll();
        
        echo json_encode([
            'success' => true,
            'message' => 'Tabla creada exitosamente con ' . $columnas['total_columnas'] . ' columnas',
            'tabla' => $nombreTabla,
            'total_columnas' => $columnas['total_columnas'],
            'estructura' => $estructura,
            'detalles' => [
                'nombre_tabla' => $nombreTabla,
                'base_datos' => $dbName,
                'servidor' => $dbHost,
                'columnas_basicas' => 1, // solo id
                'columnas_cursos' => 10, // c001 hasta c010
                'total_columnas' => $columnas['total_columnas'],
                'engine' => 'InnoDB',
                'charset' => 'utf8mb4',
                'collation' => 'utf8mb4_unicode_ci'
            ],
            'timestamp' => date('Y-m-d H:i:s')
        ]);
        
    } else {
        logCreacion("Error: Tabla no se creó después de ejecutar SQL - $nombreTabla");
        
        echo json_encode([
            'success' => false,
            'message' => 'Error: La tabla no se creó después de ejecutar el SQL',
            'timestamp' => date('Y-m-d H:i:s')
        ]);
    }
    
    // Cerrar conexión
    $pdo = null;
    logCreacion("Conexión cerrada");
    
} catch (PDOException $e) {
    // Error de conexión o SQL
    logCreacion("Error de PDO: " . $e->getMessage());
    
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Error de base de datos',
        'message' => $e->getMessage(),
        'timestamp' => date('Y-m-d H:i:s')
    ]);
    
} catch (Exception $e) {
    // Error general
    logCreacion("Error general: " . $e->getMessage());
    
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Error interno del servidor',
        'message' => $e->getMessage(),
        'timestamp' => date('Y-m-d H:i:s')
    ]);
}

logCreacion("=== FIN DE CREACIÓN DE TABLA ===");
?> 