<?php
// Configurar headers para CORS y JSON
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST');
header('Access-Control-Allow-Headers: Content-Type');
header('Content-Type: application/json; charset=utf-8');

// Habilitar reporte de errores para debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Función para logging
function logEditarRolTramos($mensaje) {
    $logFile = 'logs/editar_rol_tramos.log';
    $timestamp = date('Y-m-d H:i:s');
    $logMessage = "[$timestamp] $mensaje\n";
    
    // Crear directorio de logs si no existe
    if (!is_dir('logs')) {
        mkdir('logs', 0755, true);
    }
    
    file_put_contents($logFile, $logMessage, FILE_APPEND | LOCK_EX);
}

// Log de la operación
logEditarRolTramos("=== INICIO: Editar Rol por Tramos ===");

try {
    // Verificar método HTTP
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        throw new Exception('Método HTTP no permitido. Solo POST.');
    }

    // Obtener y decodificar datos JSON
    $input = file_get_contents('php://input');
    if (!$input) {
        throw new Exception('No se recibieron datos en el request.');
    }

    $data = json_decode($input, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        throw new Exception('Error al decodificar JSON: ' . json_last_error_msg());
    }

    // Validar datos requeridos
    $requiredFields = ['dbHost', 'dbName', 'dbUser', 'dbPassword', 'nombreTabla', 'securityKey', 'curso', 'rol', 'idInicio', 'idFinal'];
    foreach ($requiredFields as $field) {
        if (!isset($data[$field])) {
            throw new Exception("Campo requerido faltante: {$field}");
        }
    }

    // Validar clave de seguridad
    if ($data['securityKey'] !== 'clave_secreta_123') {
        throw new Exception('Clave de seguridad inválida.');
    }

    // Validar datos específicos
    if (!in_array($data['curso'], ['c001', 'c002', 'c003', 'c004', 'c005', 'c006', 'c007', 'c008', 'c009', 'c010'])) {
        throw new Exception('Curso inválido. Debe ser c001-c010.');
    }

    if ($data['rol'] !== null && (!is_numeric($data['rol']) || $data['rol'] < 0 || $data['rol'] > 4)) {
        throw new Exception('Rol inválido. Debe ser NULL o 0-4.');
    }

    if (!is_numeric($data['idInicio']) || !is_numeric($data['idFinal']) || 
        $data['idInicio'] < 1 || $data['idFinal'] < $data['idInicio']) {
        throw new Exception('Rango de IDs inválido. idInicio debe ser >= 1 y idFinal >= idInicio.');
    }

    // Extraer variables para mayor claridad
    $dbHost = $data['dbHost'];
    $dbName = $data['dbName'];
    $dbUser = $data['dbUser'];
    $dbPassword = $data['dbPassword'];
    $dbPort = isset($data['dbPort']) ? $data['dbPort'] : '3306';
    $nombreTabla = $data['nombreTabla'];
    $curso = $data['curso'];
    $rol = $data['rol'];
    $idInicio = $data['idInicio'];
    $idFinal = $data['idFinal'];

    // Log de datos recibidos
    logEditarRolTramos("Datos recibidos: " . json_encode($data));

    // Crear conexión PDO
    $dsn = "mysql:host=$dbHost;port=$dbPort;dbname=$dbName;charset=utf8mb4";
    $options = [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        PDO::ATTR_EMULATE_PREPARES => false,
    ];
    
    $pdo = new PDO($dsn, $dbUser, $dbPassword, $options);

    logEditarRolTramos("Conexión a BD exitosa");

    // Verificar si la tabla existe
    $stmt = $pdo->prepare("SELECT COUNT(*) as existe FROM information_schema.tables WHERE table_schema = ? AND table_name = ?");
    $stmt->execute([$dbName, $nombreTabla]);
    $result = $stmt->fetch();
    
    if ($result['existe'] == 0) {
        logEditarRolTramos("Error: Tabla no existe - $nombreTabla");
        throw new Exception("La tabla '{$nombreTabla}' no existe en la base de datos.");
    }

    // Verificar si la columna del curso existe
    $stmt = $pdo->prepare("SELECT COUNT(*) as existe FROM information_schema.columns WHERE table_schema = ? AND table_name = ? AND column_name = ?");
    $stmt->execute([$dbName, $nombreTabla, $curso]);
    $result = $stmt->fetch();
    
    if ($result['existe'] == 0) {
        logEditarRolTramos("Error: Columna del curso no existe - $curso");
        throw new Exception("La columna '{$curso}' no existe en la tabla '{$nombreTabla}'.");
    }

    // Contar registros en el rango antes de la actualización
    $stmt = $pdo->prepare("SELECT COUNT(*) as total FROM `$nombreTabla` WHERE id BETWEEN ? AND ?");
    $stmt->execute([$idInicio, $idFinal]);
    $totalRegistros = $stmt->fetch()['total'];

    if ($totalRegistros == 0) {
        logEditarRolTramos("Error: No se encontraron registros en el rango $idInicio - $idFinal");
        throw new Exception("No se encontraron registros con IDs entre {$idInicio} y {$idFinal}.");
    }

    logEditarRolTramos("Registros encontrados en el rango: {$totalRegistros}");

    // Comenzar transacción
    $pdo->beginTransaction();

    try {
        // Preparar la consulta de actualización usando la misma lógica que asignar_rol_completo
        if ($rol === null) {
            $sql = "UPDATE `$nombreTabla` SET `$curso` = NULL WHERE id BETWEEN ? AND ?";
            $params = [$idInicio, $idFinal];
        } else {
            $sql = "UPDATE `$nombreTabla` SET `$curso` = ? WHERE id BETWEEN ? AND ?";
            $params = [$rol, $idInicio, $idFinal];
        }

        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        
        $registrosActualizados = $stmt->rowCount();
        
        if ($registrosActualizados == 0) {
            throw new Exception("No se actualizaron registros. Verifica que los IDs existan en la tabla.");
        }

        // Confirmar transacción
        $pdo->commit();

        logEditarRolTramos("Transacción confirmada. Registros actualizados: {$registrosActualizados}");

        // Preparar respuesta de éxito
        $response = [
            'success' => true,
            'message' => "Roles actualizados exitosamente",
            'registrosActualizados' => $registrosActualizados,
            'detalles' => [
                'curso' => $curso,
                'rol' => $rol,
                'idInicio' => $idInicio,
                'idFinal' => $idFinal,
                'totalRegistrosEnRango' => $totalRegistros
            ]
        ];

        echo json_encode($response);

    } catch (Exception $e) {
        // Revertir transacción en caso de error
        $pdo->rollBack();
        throw $e;
    }

} catch (Exception $e) {
    logEditarRolTramos("ERROR: " . $e->getMessage());
    
    // Preparar respuesta de error
    $response = [
        'success' => false,
        'message' => $e->getMessage(),
        'error' => true
    ];

    http_response_code(400);
    echo json_encode($response);
}

logEditarRolTramos("=== FIN: Editar Rol por Tramos ===");
?> 