<?php
/**
 * Eliminador de tabla de inscripción de cursos
 * Elimina la tabla de la base de datos de forma segura
 */

// Configurar headers para JSON
header('Content-Type: application/json; charset=utf-8');
header('Cache-Control: no-cache, no-store, must-revalidate');
header('Pragma: no-cache');
header('Expires: 0');

// Verificar que sea una petición POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Método no permitido']);
    exit;
}

// Obtener el contenido JSON
$input = file_get_contents('php://input');
$data = json_decode($input, true);

if (!$data) {
    http_response_code(400);
    echo json_encode(['error' => 'Datos JSON inválidos']);
    exit;
}

// Extraer datos
$dbHost = $data['dbHost'] ?? '';
$dbName = $data['dbName'] ?? '';
$dbUser = $data['dbUser'] ?? '';
$dbPassword = $data['dbPassword'] ?? '';
$dbPort = $data['dbPort'] ?? '3306';
$nombreTabla = $data['nombreTabla'] ?? '';
$securityKey = $data['securityKey'] ?? '';

// Validar datos requeridos
if (empty($dbHost) || empty($dbName) || empty($dbUser) || empty($dbPort) || empty($nombreTabla)) {
    http_response_code(400);
    echo json_encode(['error' => 'Faltan datos requeridos']);
    exit;
}

// Validar clave de seguridad
$expectedKey = 'clave_secreta_123'; // Cambiar por una clave más segura en producción
if ($securityKey !== $expectedKey) {
    http_response_code(403);
    echo json_encode(['error' => 'Clave de seguridad incorrecta']);
    exit;
}

// Función para registrar logs
function logEliminacion($message) {
    $logFile = '../logs/eliminacion_tabla_inscripcion_cursos.log';
    $timestamp = date('Y-m-d H:i:s');
    $logMessage = "[$timestamp] $message\n";
    
    // Crear directorio de logs si no existe
    if (!is_dir('../logs')) {
        mkdir('../logs', 0755, true);
    }
    
    file_put_contents($logFile, $logMessage, FILE_APPEND | LOCK_EX);
}

try {
    // Registrar intento de eliminación
    logEliminacion("Eliminación de tabla - Host: $dbHost, DB: $dbName, Tabla: $nombreTabla, Usuario: $dbUser");
    
    // Crear conexión PDO
    $dsn = "mysql:host=$dbHost;port=$dbPort;dbname=$dbName;charset=utf8mb4";
    $options = [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        PDO::ATTR_EMULATE_PREPARES => false,
    ];
    
    $pdo = new PDO($dsn, $dbUser, $dbPassword, $options);
    
    // Verificar si la tabla existe antes de eliminar
    $stmt = $pdo->prepare("SELECT COUNT(*) as existe FROM information_schema.tables WHERE table_schema = ? AND table_name = ?");
    $stmt->execute([$dbName, $nombreTabla]);
    $result = $stmt->fetch();
    
    if ($result['existe'] == 0) {
        // La tabla no existe
        logEliminacion("Error: Tabla no existe - $nombreTabla");
        
        echo json_encode([
            'success' => false,
            'message' => 'La tabla no existe en la base de datos',
            'timestamp' => date('Y-m-d H:i:s')
        ]);
        exit;
    }
    
    // Obtener información de la tabla antes de eliminar
    $stmt = $pdo->prepare("
        SELECT COUNT(*) as total_columnas
        FROM information_schema.columns 
        WHERE table_schema = ? AND table_name = ?
    ");
    $stmt->execute([$dbName, $nombreTabla]);
    $columnas = $stmt->fetch();
    
    logEliminacion("Tabla encontrada con " . $columnas['total_columnas'] . " columnas, procediendo a eliminar...");
    
    // Eliminar la tabla
    $sql = "DROP TABLE IF EXISTS `$nombreTabla`";
    $pdo->exec($sql);
    
    logEliminacion("Comando DROP TABLE ejecutado exitosamente");
    
    // Verificar que la tabla se eliminó correctamente
    $stmt = $pdo->prepare("SELECT COUNT(*) as existe FROM information_schema.tables WHERE table_schema = ? AND table_name = ?");
    $stmt->execute([$dbName, $nombreTabla]);
    $result = $stmt->fetch();
    
    if ($result['existe'] == 0) {
        // Tabla eliminada exitosamente
        logEliminacion("Tabla eliminada exitosamente - $nombreTabla");
        
        echo json_encode([
            'success' => true,
            'message' => 'Tabla eliminada exitosamente. Se eliminaron ' . $columnas['total_columnas'] . ' columnas.',
            'tabla' => $nombreTabla,
            'columnas_eliminadas' => $columnas['total_columnas'],
            'timestamp' => date('Y-m-d H:i:s')
        ]);
        
    } else {
        // Error: la tabla no se eliminó
        logEliminacion("Error: Tabla no se eliminó después de ejecutar DROP TABLE - $nombreTabla");
        
        echo json_encode([
            'success' => false,
            'message' => 'Error: La tabla no se eliminó después de ejecutar el comando DROP TABLE',
            'timestamp' => date('Y-m-d H:i:s')
        ]);
    }
    
    // Cerrar conexión
    $pdo = null;
    
} catch (PDOException $e) {
    // Error de conexión o SQL
    logEliminacion("Error de PDO: " . $e->getMessage());
    
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Error de base de datos',
        'message' => $e->getMessage(),
        'timestamp' => date('Y-m-d H:i:s')
    ]);
    
} catch (Exception $e) {
    // Error general
    logEliminacion("Error general: " . $e->getMessage());
    
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Error interno del servidor',
        'message' => $e->getMessage(),
        'timestamp' => date('Y-m-d H:i:s')
    ]);
}

logEliminacion("=== FIN DE ELIMINACIÓN DE TABLA ===");
?> 