<?php
/**
 * Listador de tablas existentes en base de datos
 * Lista todas las tablas disponibles en una base de datos específica
 */

// Configurar headers para JSON
header('Content-Type: application/json; charset=utf-8');
header('Cache-Control: no-cache, no-store, must-revalidate');
header('Pragma: no-cache');
header('Expires: 0');

// Verificar que sea una petición POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Método no permitido']);
    exit;
}

// Obtener el contenido JSON
$input = file_get_contents('php://input');
$data = json_decode($input, true);

if (!$data) {
    http_response_code(400);
    echo json_encode(['error' => 'Datos JSON inválidos']);
    exit;
}

// Extraer datos
$dbHost = $data['dbHost'] ?? '';
$dbName = $data['dbName'] ?? '';
$dbUser = $data['dbUser'] ?? '';
$dbPassword = $data['dbPassword'] ?? '';
$dbPort = $data['dbPort'] ?? '3306';
$securityKey = $data['securityKey'] ?? '';

// Validar datos requeridos
if (empty($dbHost) || empty($dbName) || empty($dbUser) || empty($dbPort)) {
    http_response_code(400);
    echo json_encode(['error' => 'Faltan datos requeridos']);
    exit;
}

// Validar clave de seguridad
$expectedKey = 'clave_secreta_123'; // Cambiar por una clave más segura en producción
if ($securityKey !== $expectedKey) {
    http_response_code(403);
    echo json_encode(['error' => 'Clave de seguridad incorrecta']);
    exit;
}

// Función para registrar logs
function logListado($message) {
    $logFile = '../logs/listado_tablas_bd.log';
    $timestamp = date('Y-m-d H:i:s');
    $logMessage = "[$timestamp] $message\n";
    
    // Crear directorio de logs si no existe
    if (!is_dir('../logs')) {
        mkdir('../logs', 0755, true);
    }
    
    file_put_contents($logFile, $logMessage, FILE_APPEND | LOCK_EX);
}

try {
    // Registrar intento de listado
    logListado("Listado de tablas - Host: $dbHost, DB: $dbName, Usuario: $dbUser");
    
    // Crear conexión PDO
    $dsn = "mysql:host=$dbHost;port=$dbPort;dbname=$dbName;charset=utf8mb4";
    $options = [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        PDO::ATTR_EMULATE_PREPARES => false,
    ];
    
    $pdo = new PDO($dsn, $dbUser, $dbPassword, $options);
    
    // Obtener lista de todas las tablas
    $stmt = $pdo->prepare("
        SELECT TABLE_NAME as nombre
        FROM information_schema.tables 
        WHERE table_schema = ?
        ORDER BY TABLE_NAME
    ");
    $stmt->execute([$dbName]);
    $tablas = $stmt->fetchAll();
    
    // Extraer solo los nombres de las tablas
    $nombresTablas = array_column($tablas, 'nombre');
    
    logListado("Listado exitoso - Se encontraron " . count($nombresTablas) . " tablas");
    
    echo json_encode([
        'success' => true,
        'tablas' => $nombresTablas,
        'total' => count($nombresTablas),
        'message' => 'Listado de tablas obtenido exitosamente',
        'timestamp' => date('Y-m-d H:i:s')
    ]);
    
    // Cerrar conexión
    $pdo = null;
    
} catch (PDOException $e) {
    // Error de conexión
    logListado("Error de conexión: " . $e->getMessage());
    
    http_response_code(500);
    echo json_encode([
        'error' => 'Error de conexión a la base de datos',
        'message' => $e->getMessage(),
        'timestamp' => date('Y-m-d H:i:s')
    ]);
    
} catch (Exception $e) {
    // Error general
    logListado("Error general: " . $e->getMessage());
    
    http_response_code(500);
    echo json_encode([
        'error' => 'Error interno del servidor',
        'message' => $e->getMessage(),
        'timestamp' => date('Y-m-d H:i:s')
    ]);
}
?> 