<?php
/**
 * Verificador de existencia de tabla de inscripción de cursos
 * Verifica si una tabla existe en la base de datos y obtiene información detallada
 */

// Configurar headers para JSON
header('Content-Type: application/json; charset=utf-8');
header('Cache-Control: no-cache, no-store, must-revalidate');
header('Pragma: no-cache');
header('Expires: 0');

// Verificar que sea una petición POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Método no permitido']);
    exit;
}

// Obtener el contenido JSON
$input = file_get_contents('php://input');
$data = json_decode($input, true);

if (!$data) {
    http_response_code(400);
    echo json_encode(['error' => 'Datos JSON inválidos']);
    exit;
}

// Extraer datos
$dbHost = $data['dbHost'] ?? '';
$dbName = $data['dbName'] ?? '';
$dbUser = $data['dbUser'] ?? '';
$dbPassword = $data['dbPassword'] ?? '';
$dbPort = $data['dbPort'] ?? '3306';
$nombreTabla = $data['nombreTabla'] ?? '';
$securityKey = $data['securityKey'] ?? '';

// Validar datos requeridos
if (empty($dbHost) || empty($dbName) || empty($dbUser) || empty($dbPort) || empty($nombreTabla)) {
    http_response_code(400);
    echo json_encode(['error' => 'Faltan datos requeridos']);
    exit;
}

// Validar clave de seguridad
$expectedKey = 'clave_secreta_123'; // Cambiar por una clave más segura en producción
if ($securityKey !== $expectedKey) {
    http_response_code(403);
    echo json_encode(['error' => 'Clave de seguridad incorrecta']);
    exit;
}

// Función para registrar logs
function logVerification($message) {
    $logFile = '../logs/verificacion_tabla_inscripcion_cursos.log';
    $timestamp = date('Y-m-d H:i:s');
    $logMessage = "[$timestamp] $message\n";
    
    // Crear directorio de logs si no existe
    if (!is_dir('../logs')) {
        mkdir('../logs', 0755, true);
    }
    
    file_put_contents($logFile, $logMessage, FILE_APPEND | LOCK_EX);
}

try {
    // Registrar intento de verificación
    logVerification("Verificación de tabla - Host: $dbHost, DB: $dbName, Tabla: $nombreTabla, Usuario: $dbUser");
    
    // Crear conexión PDO
    $dsn = "mysql:host=$dbHost;port=$dbPort;dbname=$dbName;charset=utf8mb4";
    $options = [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        PDO::ATTR_EMULATE_PREPARES => false,
    ];
    
    $pdo = new PDO($dsn, $dbUser, $dbPassword, $options);
    
    // Obtener lista de todas las tablas existentes
    $stmt = $pdo->prepare("SELECT TABLE_NAME as nombre FROM information_schema.tables WHERE table_schema = ? ORDER BY TABLE_NAME");
    $stmt->execute([$dbName]);
    $todasTablas = $stmt->fetchAll();
    $nombresTablas = array_column($todasTablas, 'nombre');
    $tablasExistentes = implode(', ', $nombresTablas);
    
    // Obtener información detallada de TODAS las tablas existentes
    $informacionTodasTablas = [];
    foreach ($nombresTablas as $nombreTablaExistente) {
        $informacionTodasTablas[$nombreTablaExistente] = obtenerInformacionTabla($pdo, $dbName, $nombreTablaExistente);
    }
    
    // Verificar si la tabla específica existe
    $stmt = $pdo->prepare("SELECT COUNT(*) as existe FROM information_schema.tables WHERE table_schema = ? AND table_name = ?");
    $stmt->execute([$dbName, $nombreTabla]);
    $result = $stmt->fetch();
    
    $tablaExiste = $result['existe'] > 0;
    
    if ($tablaExiste) {
        // Obtener información detallada de la tabla objetivo
        $informacion = obtenerInformacionTabla($pdo, $dbName, $nombreTabla);
        
        logVerification("Tabla encontrada - Información obtenida exitosamente");
        
        echo json_encode([
            'success' => true,
            'tablaExiste' => true,
            'informacion' => $informacion,
            'tablasExistentes' => $tablasExistentes,
            'informacionTodasTablas' => $informacionTodasTablas,
            'message' => 'Tabla encontrada en la base de datos',
            'timestamp' => date('Y-m-d H:i:s')
        ]);
        
    } else {
        // Tabla no existe
        logVerification("Tabla no encontrada - Lista para crear");
        
        echo json_encode([
            'success' => true,
            'tablaExiste' => false,
            'tablasExistentes' => $tablasExistentes,
            'informacionTodasTablas' => $informacionTodasTablas,
            'message' => 'Tabla no existe, puede proceder a crearla',
            'timestamp' => date('Y-m-d H:i:s')
        ]);
    }
    
    // Cerrar conexión
    $pdo = null;
    
} catch (PDOException $e) {
    // Error de conexión
    logVerification("Error de conexión: " . $e->getMessage());
    
    http_response_code(500);
    echo json_encode([
        'error' => 'Error de conexión a la base de datos',
        'message' => $e->getMessage(),
        'timestamp' => date('Y-m-d H:i:s')
    ]);
    
} catch (Exception $e) {
    // Error general
    logVerification("Error general: " . $e->getMessage());
    
    http_response_code(500);
    echo json_encode([
        'error' => 'Error interno del servidor',
        'message' => $e->getMessage(),
        'timestamp' => date('Y-m-d H:i:s')
    ]);
}

/**
 * Obtiene información detallada de una tabla existente
 */
function obtenerInformacionTabla($pdo, $dbName, $nombreTabla) {
    $informacion = [];
    
    try {
        // Información básica de la tabla
        $stmt = $pdo->prepare("
            SELECT 
                TABLE_NAME as nombre,
                ENGINE as engine,
                TABLE_COLLATION as collation,
                TABLE_COMMENT as comment,
                CREATE_TIME as fechaCreacion
            FROM information_schema.tables 
            WHERE table_schema = ? AND table_name = ?
        ");
        $stmt->execute([$dbName, $nombreTabla]);
        $tabla = $stmt->fetch();
        
        if ($tabla) {
            $informacion['nombre'] = $tabla['nombre'];
            $informacion['engine'] = $tabla['engine'];
            $informacion['collation'] = $tabla['collation'];
            $informacion['comment'] = $tabla['comment'];
            $informacion['fechaCreacion'] = $tabla['fechaCreacion'] ? date('Y-m-d H:i:s', strtotime($tabla['fechaCreacion'])) : 'No disponible';
        }
        
        // Contar columnas
        $stmt = $pdo->prepare("
            SELECT COUNT(*) as total
            FROM information_schema.columns 
            WHERE table_schema = ? AND table_name = ?
        ");
        $stmt->execute([$dbName, $nombreTabla]);
        $columnas = $stmt->fetch();
        $informacion['columnas'] = $columnas['total'];
        
        // Contar índices
        $stmt = $pdo->prepare("
            SELECT COUNT(*) as total
            FROM information_schema.statistics 
            WHERE table_schema = ? AND table_name = ?
        ");
        $stmt->execute([$dbName, $nombreTabla]);
        $indices = $stmt->fetch();
        $informacion['indices'] = $indices['total'];
        
        // Obtener columnas de cursos (c001, c002, etc.) si es la tabla objetivo
        if ($nombreTabla === 'inscripcion_cursos') {
            $stmt = $pdo->prepare("
                SELECT COLUMN_NAME
                FROM information_schema.columns 
                WHERE table_schema = ? AND table_name = ? AND COLUMN_NAME LIKE 'c%'
                ORDER BY COLUMN_NAME
            ");
            $stmt->execute([$dbName, $nombreTabla]);
            $columnasCursos = $stmt->fetchAll();
            
            $informacion['columnasCursos'] = array_column($columnasCursos, 'COLUMN_NAME');
            $informacion['totalCursos'] = count($informacion['columnasCursos']);
            
            // Verificar si tiene la estructura esperada
            $informacion['estructuraCorrecta'] = verificarEstructuraEsperada($informacion['columnasCursos']);
        }
        
        // Obtener información detallada de cada columna
        $stmt = $pdo->prepare("
            SELECT 
                COLUMN_NAME,
                DATA_TYPE,
                IS_NULLABLE,
                COLUMN_DEFAULT,
                COLUMN_COMMENT,
                ORDINAL_POSITION
            FROM information_schema.columns 
            WHERE table_schema = ? AND table_name = ?
            ORDER BY ORDINAL_POSITION
        ");
        $stmt->execute([$dbName, $nombreTabla]);
        $detalleColumnas = $stmt->fetchAll();
        $informacion['detalleColumnas'] = $detalleColumnas;
        
        // Contar tipos de columnas
        $tiposColumnas = [];
        foreach ($detalleColumnas as $col) {
            $tipo = $col['DATA_TYPE'];
            $tiposColumnas[$tipo] = ($tiposColumnas[$tipo] ?? 0) + 1;
        }
        $informacion['tiposColumnas'] = $tiposColumnas;
        
        // Contar columnas por categoría (solo para tabla objetivo)
        if ($nombreTabla === 'inscripcion_cursos') {
            $informacion['columnasBasicas'] = 0;
            $informacion['columnasCursos'] = 0;
            $informacion['columnasOtros'] = 0;
            
            foreach ($detalleColumnas as $col) {
                $nombre = $col['COLUMN_NAME'];
                if (in_array($nombre, ['id', 'usuario_id', 'nombre_usuario', 'email_usuario', 'fecha_inscripcion', 'estado'])) {
                    $informacion['columnasBasicas']++;
                } elseif (preg_match('/^c\d{3}$/', $nombre)) {
                    $informacion['columnasCursos']++;
                } else {
                    $informacion['columnasOtros']++;
                }
            }
        }
        
    } catch (Exception $e) {
        // Si hay error al obtener información detallada, usar información básica
        $informacion['error'] = 'Error al obtener información detallada: ' . $e->getMessage();
    }
    
    return $informacion;
}

/**
 * Verifica si la estructura de la tabla coincide con lo esperado
 */
function verificarEstructuraEsperada($columnasCursos) {
    if (empty($columnasCursos)) {
        return false;
    }
    
    // Verificar que las columnas sigan el patrón c001, c002, c003...
    $patron = '/^c\d{3}$/';
    $estructuraCorrecta = true;
    
    foreach ($columnasCursos as $columna) {
        if (!preg_match($patron, $columna)) {
            $estructuraCorrecta = false;
            break;
        }
    }
    
    return $estructuraCorrecta;
}
?> 